# 🚀 Deployment Guide for medi8

## Table of Contents
1. [Prerequisites](#prerequisites)
2. [Local Development Setup](#local-development-setup)
3. [Production Deployment](#production-deployment)
4. [Environment Configuration](#environment-configuration)
5. [Database Setup](#database-setup)
6. [Troubleshooting](#troubleshooting)

---

## Prerequisites

### Required Software
- **Node.js** 18.x or higher ([Download](https://nodejs.org/))
- **npm** 9.x or higher (comes with Node.js)
- **PostgreSQL** 14.x or higher ([Download](https://www.postgresql.org/download/))
- **Git** (optional but recommended)

### Recommended Tools
- **VS Code** or your preferred editor
- **pgAdmin** or **TablePlus** for database management
- **Postman** for API testing (future milestones)

---

## Local Development Setup

### Step 1: Download the Code

**From Claude outputs:**
1. Download the `medi8` folder
2. Extract to your preferred location
3. Open terminal in the `medi8` directory

**Or clone from Git:**
```bash
git clone <your-repository-url>
cd medi8
```

### Step 2: Install Dependencies

```bash
npm install
```

This will install:
- Next.js 14
- React 18
- Prisma ORM
- TypeScript
- Tailwind CSS
- All other dependencies

**Expected time:** 2-3 minutes

### Step 3: Configure Environment

```bash
# Copy the example environment file
cp .env.example .env.local

# Open for editing
# On Mac/Linux:
nano .env.local
# On Windows:
notepad .env.local
```

**Minimum required configuration:**
```bash
DATABASE_URL="postgresql://postgres:password@localhost:5432/medi8"
APP_URL="http://localhost:3000"
NODE_ENV="development"
```

### Step 4: Set Up PostgreSQL

**Option A: Local PostgreSQL**
```bash
# Create database
psql -U postgres
CREATE DATABASE medi8;
\q
```

**Option B: Docker PostgreSQL**
```bash
docker run --name medi8-postgres \
  -e POSTGRES_PASSWORD=password \
  -e POSTGRES_DB=medi8 \
  -p 5432:5432 \
  -d postgres:14

# Verify it's running
docker ps
```

**Option C: Cloud Database**
- [Neon](https://neon.tech/) - Free PostgreSQL
- [Supabase](https://supabase.com/) - Free tier with 500MB
- [Railway](https://railway.app/) - Easy PostgreSQL hosting

Get connection string and update `DATABASE_URL` in `.env.local`

### Step 5: Initialize Database

```bash
# Generate Prisma Client
npm run db:generate

# Run migrations (creates all tables)
npm run db:migrate

# Seed initial data
npm run db:seed
```

**What gets seeded:**
- Admin user: `admin@medi8.com.au` / `admin123`
- Platform configuration (fees, trial period)
- Email notification templates

### Step 6: Start Development Server

```bash
npm run dev
```

Open browser to: **http://localhost:3000**

**You should see:**
- Homepage with hero section
- Login and registration links
- Professional design

### Step 7: Verify Setup

**Test admin login:**
1. Go to http://localhost:3000/login
2. Login with: `admin@medi8.com.au` / `admin123`
3. Should redirect to admin dashboard

**Test mediator registration:**
1. Go to http://localhost:3000/register/mediator
2. Fill in the form
3. Should redirect to mediator dashboard

**Test client registration:**
1. Go to http://localhost:3000/register/client
2. Fill in the form
3. Should redirect to client dashboard

✅ **Setup complete!** Your local development environment is ready.

---

## Production Deployment

### Option 1: Vercel (Recommended - Easiest)

**Why Vercel?**
- Zero configuration for Next.js
- Automatic HTTPS
- Global CDN
- Free SSL certificates
- Automatic deployments on Git push

**Steps:**

1. **Push code to GitHub**
```bash
git init
git add .
git commit -m "Initial commit"
git remote add origin <your-github-repo>
git push -u origin main
```

2. **Deploy to Vercel**
- Go to [vercel.com](https://vercel.com)
- Click "Import Project"
- Select your GitHub repository
- Vercel auto-detects Next.js settings

3. **Add Environment Variables**

In Vercel dashboard, add:
```
DATABASE_URL=<your-production-database-url>
APP_URL=<your-vercel-url>
NODE_ENV=production
```

4. **Add PostgreSQL Database**

**Option A: Vercel Postgres**
- Go to Storage tab in Vercel
- Create Postgres database
- Connection string automatically added

**Option B: External Database**
- Use Neon, Supabase, or Railway
- Copy connection string to `DATABASE_URL`

5. **Run Migrations**

In Vercel, add build command:
```bash
# In Project Settings > Build & Development Settings
Build Command: npm run build && npx prisma migrate deploy
```

Or run manually:
```bash
# Install Vercel CLI
npm i -g vercel

# Login
vercel login

# Run migration
vercel env pull .env.local
npm run db:migrate:prod
npm run db:seed
```

6. **Deploy!**

Vercel deploys automatically on every push to main branch.

**Your app will be live at:** `https://your-project.vercel.app`

### Option 2: Railway

**Why Railway?**
- Simple deployment
- Built-in PostgreSQL
- Automatic HTTPS
- Pay-as-you-grow pricing

**Steps:**

1. **Create Railway Account**
- Go to [railway.app](https://railway.app)
- Sign up with GitHub

2. **Create New Project**
- Click "New Project"
- Select "Deploy from GitHub repo"
- Choose your `medi8` repository

3. **Add PostgreSQL**
- Click "New" → "Database" → "PostgreSQL"
- Railway automatically adds `DATABASE_URL`

4. **Configure Environment**
- Go to Variables tab
- Add:
  ```
  APP_URL=<your-railway-url>
  NODE_ENV=production
  ```

5. **Add Build Command**
```
npm run db:generate && npm run db:migrate:prod && npm run build
```

6. **Add Start Command**
```
npm start
```

7. **Deploy**
- Railway deploys automatically
- Run seed script in terminal:
```bash
railway run npm run db:seed
```

**Your app will be live at:** Railway-provided URL

### Option 3: VPS (DigitalOcean, AWS, Linode)

**For experienced users who want full control**

**Requirements:**
- Ubuntu 22.04 or similar
- At least 1GB RAM
- Node.js 18+
- PostgreSQL 14+
- Nginx (reverse proxy)

**Steps:**

1. **SSH into your server**
```bash
ssh root@your-server-ip
```

2. **Install Node.js**
```bash
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs
```

3. **Install PostgreSQL**
```bash
sudo apt update
sudo apt install postgresql postgresql-contrib
```

4. **Create database**
```bash
sudo -u postgres psql
CREATE DATABASE medi8;
CREATE USER medi8user WITH PASSWORD 'your-secure-password';
GRANT ALL PRIVILEGES ON DATABASE medi8 TO medi8user;
\q
```

5. **Clone your repository**
```bash
cd /var/www
git clone <your-repo-url>
cd medi8
```

6. **Install dependencies**
```bash
npm install
```

7. **Configure environment**
```bash
nano .env.local
```

Add:
```
DATABASE_URL="postgresql://medi8user:your-password@localhost:5432/medi8"
APP_URL="https://yourdomain.com"
NODE_ENV="production"
```

8. **Set up database**
```bash
npm run db:generate
npm run db:migrate:prod
npm run db:seed
```

9. **Build application**
```bash
npm run build
```

10. **Install PM2**
```bash
npm install -g pm2
pm2 start npm --name "medi8" -- start
pm2 save
pm2 startup
```

11. **Configure Nginx**
```bash
sudo nano /etc/nginx/sites-available/medi8
```

Add:
```nginx
server {
    listen 80;
    server_name yourdomain.com;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
```

Enable site:
```bash
sudo ln -s /etc/nginx/sites-available/medi8 /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

12. **Set up SSL with Let's Encrypt**
```bash
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d yourdomain.com
```

✅ **Production deployment complete!**

---

## Environment Configuration

### Development (.env.local)

```bash
# Database
DATABASE_URL="postgresql://postgres:password@localhost:5432/medi8"

# App
APP_URL="http://localhost:3000"
NODE_ENV="development"

# Storage (local for development)
STORAGE_PROVIDER="local"

# Email (console for development)
EMAIL_PROVIDER="console"
```

### Production (.env.production)

```bash
# Database (use your production database)
DATABASE_URL="postgresql://user:password@host:5432/medi8?sslmode=require"

# App (your actual domain)
APP_URL="https://yourdomain.com"
NODE_ENV="production"

# Stripe (when ready for payments)
STRIPE_SECRET_KEY="sk_live_..."
STRIPE_PUBLISHABLE_KEY="pk_live_..."
STRIPE_WEBHOOK_SECRET="whsec_..."

# Storage (S3 for production - optional)
STORAGE_PROVIDER="s3"
AWS_ACCESS_KEY_ID="..."
AWS_SECRET_ACCESS_KEY="..."
AWS_S3_BUCKET="medi8-uploads"
AWS_REGION="ap-southeast-2"

# Email (SendGrid for production - optional)
EMAIL_PROVIDER="sendgrid"
SENDGRID_API_KEY="SG...."
```

---

## Database Setup

### Migrations

**Create new migration:**
```bash
npm run db:migrate
```

**Deploy to production:**
```bash
npm run db:migrate:prod
```

**Reset database (CAREFUL!):**
```bash
npm run db:reset  # Deletes all data
```

### Prisma Studio

Visual database browser:
```bash
npm run db:studio
```

Opens at: http://localhost:5555

### Database Backups

**PostgreSQL backup:**
```bash
pg_dump -U postgres medi8 > backup.sql
```

**Restore:**
```bash
psql -U postgres medi8 < backup.sql
```

---

## Troubleshooting

### Common Issues

**1. "Cannot find module '@prisma/client'"**

**Solution:**
```bash
npm run db:generate
```

**2. "P1001: Can't reach database server"**

**Causes:**
- PostgreSQL not running
- Wrong connection string
- Firewall blocking port 5432

**Solution:**
```bash
# Check if PostgreSQL is running
pg_isready

# Check connection string
echo $DATABASE_URL

# Test connection
psql $DATABASE_URL
```

**3. "P3009: migrate found failed migration"**

**Solution:**
```bash
# Mark migration as applied
npx prisma migrate resolve --applied <migration-name>

# Or reset (WARNING: deletes data)
npm run db:reset
```

**4. "Port 3000 already in use"**

**Solution:**
```bash
# Kill process on port 3000
lsof -ti:3000 | xargs kill -9

# Or use different port
PORT=3001 npm run dev
```

**5. "Session not persisting"**

**Causes:**
- Cookies not enabled
- Wrong APP_URL
- HTTPS/HTTP mismatch

**Solution:**
- Clear browser cookies
- Check APP_URL matches your domain
- Ensure consistent protocol (http:// or https://)

**6. "Build fails on Vercel"**

**Common causes:**
- TypeScript errors
- Missing environment variables
- Build command incorrect

**Solution:**
```bash
# Test build locally
npm run build

# Check for TypeScript errors
npx tsc --noEmit

# Verify environment variables in Vercel dashboard
```

### Getting Help

**Check logs:**
```bash
# Development
npm run dev

# Production with PM2
pm2 logs medi8

# Vercel logs
vercel logs
```

**Database inspection:**
```bash
npm run db:studio
```

**Community:**
- Next.js: https://nextjs.org/docs
- Prisma: https://www.prisma.io/docs
- Vercel: https://vercel.com/docs

---

## Post-Deployment Checklist

After deploying to production:

- [ ] **Test all user flows**
  - [ ] Admin login
  - [ ] Mediator registration
  - [ ] Client registration
  - [ ] Profile editing
  - [ ] Admin approvals
  - [ ] Public practice pages

- [ ] **Security**
  - [ ] Change admin password
  - [ ] Set secure DATABASE_URL
  - [ ] Enable HTTPS
  - [ ] Configure CORS if needed

- [ ] **Performance**
  - [ ] Test page load times
  - [ ] Check database queries (Prisma Studio)
  - [ ] Monitor error logs

- [ ] **Backups**
  - [ ] Set up automatic database backups
  - [ ] Document restore procedure
  - [ ] Test backup restore process

- [ ] **Monitoring**
  - [ ] Set up error tracking (Sentry, etc.)
  - [ ] Configure uptime monitoring
  - [ ] Set up alerts

---

## Next Steps

After successful deployment:

1. **Customize branding**
   - Update logo and colors
   - Add your company information
   - Customize email templates

2. **Configure payments (Milestone 5)**
   - Set up Stripe account
   - Add Stripe keys to environment
   - Test payment flow

3. **Enable email notifications (Milestone 6)**
   - Set up SendGrid account
   - Add API key to environment
   - Test email delivery

4. **Launch!**
   - Invite beta testers
   - Gather feedback
   - Iterate and improve

---

**Need help? Check README.md for detailed information or MILESTONE_2_SUMMARY.md for feature documentation.**
