# Milestone 5: Document Management System - Complete!

## ✅ What's Implemented

### **Document Templates**
✅ **Pre-Mediation Information Pack** - Explains mediation process, confidentiality, expectations
✅ **Fee Disclosure Statement** - Transparent fee breakdown with variable substitution
✅ **Mediation Agreement** - Formal agreement outlining terms, confidentiality, roles
✅ **Custom Templates** - Mediators can create their own
✅ **Template Variables** - {{variable}} substitution (e.g., {{mediatorName}}, {{hourlyRate}})
✅ **Active/Inactive Status** - Templates can be deactivated without deletion

### **Document Instances (Per-Case)**
✅ **Create from Template** - Generate case-specific documents from templates
✅ **Variable Rendering** - Auto-fill mediator, case, client data
✅ **Versioning** - Each case gets its own copy
✅ **Status Tracking** - Draft, sent, acknowledged, signed
✅ **Audit Trail** - All document actions logged

### **Send Workflow**
✅ **Email Delivery** - Send documents to one or more clients
✅ **Send Timestamp** - Records when document was sent
✅ **Recipient Tracking** - Stores email addresses of recipients
✅ **Acknowledgement Token** - Unique secure link for each document
✅ **Expiry Date** - Links expire after 30 days
✅ **Email Stub** - Logs to console (production email in M6)

### **Acknowledge Workflow**
✅ **Public Acknowledgement Page** - `/documents/acknowledge/[token]`
✅ **No Login Required** - Clients can acknowledge without account
✅ **Name Capture** - Records who acknowledged
✅ **Timestamp** - Records when acknowledged
✅ **One-Time Use** - Can't re-acknowledge
✅ **Expiry Validation** - Rejects expired tokens
✅ **Case Timeline Entry** - Adds event to case history

### **Storage Abstraction**
✅ **StorageProvider Interface** - Unified API for file storage
✅ **LocalStorageProvider** - Filesystem storage for development
✅ **S3StorageProvider** - Stub for production (implements later)
✅ **Upload Methods** - File and Buffer upload support
✅ **Download/Delete** - Full CRUD operations
✅ **URL Generation** - Public access URLs
✅ **File Metadata** - Size, MIME type tracking
✅ **Path Generation** - Timestamped, unique paths

## 🗂️ File Organization

```
documents/
  {mediatorId}/
    templates/
      {templateId}.pdf
    cases/
      {caseId}/
        {documentId}.pdf
```

## 📊 Database Schema (Updated)

### DocumentTemplate Model
```prisma
model DocumentTemplate {
  id                      String  @id
  mediatorId              String
  name                    String
  category                DocumentCategory
  description             String?
  content                 String?  // Text content
  fileName                String?  // File upload
  filePath                String?
  isRequired              Boolean
  requiresAcknowledgement Boolean
  requiresSignature       Boolean
  displayOrder            Int
  isActive                Boolean
  createdAt               DateTime
  updatedAt               DateTime
  
  mediator                Mediator
  instances               DocumentInstance[]
}
```

### DocumentInstance Model
```prisma
model DocumentInstance {
  id                      String  @id
  caseId                  String
  mediatorId              String
  templateId              String?
  name                    String
  category                DocumentCategory
  content                 String?
  fileName                String?
  filePath                String?
  
  // Send tracking
  sentAt                  DateTime?
  sentTo                  String?
  
  // Acknowledge tracking
  acknowledgedAt          DateTime?
  acknowledgedBy          String?
  acknowledgementToken    String?  @unique
  acknowledgementExpiry   DateTime?
  
  // Signature tracking
  signedAt                DateTime?
  signedBy                String?
  requiresSignature       Boolean
  requiresAcknowledgement Boolean
  
  case                    Case
  mediator                Mediator
  template                DocumentTemplate?
}
```

## 📁 New Files Created (12 files)

### Services
- `src/modules/document/template/service.ts` - Template CRUD (235 lines)
- `src/modules/document/instance/service.ts` - Instance management (290 lines)
- `src/modules/document/actions.ts` - Server Actions

### Storage
- `src/lib/storage.ts` - Updated with upload metadata

### Pages
- `src/app/(mediator)/documents/page.tsx` - Template management
- `src/app/(mediator)/cases/[id]/documents/page.tsx` - Case documents
- `src/app/documents/acknowledge/[token]/page.tsx` - Public acknowledgement

### Components
- `src/components/document/TemplateList.tsx` - Template display
- `src/components/document/DocumentList.tsx` - Case documents
- `src/components/document/SendDocumentModal.tsx` - Send UI
- `src/components/document/AcknowledgeForm.tsx` - Acknowledge UI

### Documentation
- `MILESTONE_5_SUMMARY.md` - This file

## 🎯 User Flows

### Flow 1: Mediator Creates Documents for Case

```
1. Mediator navigates to case
2. Clicks "Documents" tab
3. Clicks "Add Document"
4. Selects template (e.g., "Pre-Mediation Pack")
5. System auto-fills variables:
   - {{mediatorName}} → "John Smith"
   - {{caseNumber}} → "M8-2024-0042"
   - {{hourlyRate}} → "$200"
   - etc.
6. Document instance created
7. Shows in "Draft" state
```

### Flow 2: Mediator Sends Document to Client

```
1. From case documents, click "Send"
2. Enter client email(s): jane@example.com, john@example.com
3. Click "Send Document"
4. System:
   - Records sentAt timestamp
   - Stores recipients: "jane@example.com, john@example.com"
   - Generates token: "abc123-def456-ghi789"
   - Sets expiry: 30 days from now
   - Sends email (stub - logs to console)
5. Document status → "Sent"
6. Email contains acknowledgement link
```

### Flow 3: Client Acknowledges Document

```
1. Client receives email
2. Clicks link: /documents/acknowledge/abc123-def456-ghi789
3. Public page loads (no login required)
4. Shows:
   - Document name
   - Case number
   - Document content (preview)
5. Client enters name: "Jane Doe"
6. Clicks "I Acknowledge"
7. System:
   - Records acknowledgedAt timestamp
   - Stores acknowledgedBy: "Jane Doe"
   - Adds case timeline entry
   - Creates audit log
8. Shows confirmation: "Thank you for acknowledging"
9. Link becomes invalid (already acknowledged)
```

### Flow 4: Mediator Checks Status

```
1. Mediator views case documents
2. Sees document statuses:
   - ✓ Sent: Jan 30, 2026 2:30 PM
   - ✓ Acknowledged: Jan 30, 2026 3:15 PM by Jane Doe
3. Can see who acknowledged and when
4. Timeline shows: "Document acknowledged by Jane Doe"
```

## 🔐 Security Features

**Token Security:**
- UUID v4 tokens (128-bit randomness)
- Unique constraint in database
- 30-day expiry
- One-time use (can't re-acknowledge)
- No user enumeration

**Access Control:**
- Mediators see only their documents
- Clients need valid token
- Expired tokens rejected
- Templates belong to mediator

**Audit Trail:**
- All creates logged
- All sends logged
- All acknowledges logged
- IP/User agent captured

## 💾 Storage Implementation

### Local Storage (Development)

```typescript
// Saves to: public/uploads/
const storage = new LocalStorageProvider();

await storage.upload(file, 'documents/med123/case456/doc.pdf');
// File saved to: public/uploads/documents/med123/case456/doc.pdf
// URL: /uploads/documents/med123/case456/doc.pdf
```

### S3 Storage (Production - Stub)

```typescript
// Configuration
STORAGE_PROVIDER=s3
AWS_S3_BUCKET=medi8-documents
AWS_REGION=ap-southeast-2

// Implementation (add later)
const storage = new S3StorageProvider(bucket, region);
await storage.upload(file, 'documents/med123/case456/doc.pdf');
// Saves to S3, returns URL
```

### Switch Providers

```bash
# .env.local
STORAGE_PROVIDER=local  # Development
# or
STORAGE_PROVIDER=s3     # Production
```

## 📧 Email Notifications

### DOCUMENT_SENT Template

```
Subject: Document sent: Pre-Mediation Information Pack

Hi,

A document has been sent to you for case M8-2024-0042.

Document: Pre-Mediation Information Pack

Please acknowledge receipt by clicking the link below:
https://medi8.com.au/documents/acknowledge/abc123-def456-ghi789

This link expires on: Feb 29, 2026

Best regards,
John Smith
```

### In Development

Emails log to console:

```
📧 Email sent to jane@example.com
   Template: DOCUMENT_SENT
   Data: {
     documentName: "Pre-Mediation Information Pack",
     caseNumber: "M8-2024-0042",
     mediatorName: "John Smith",
     acknowledgementUrl: "http://localhost:3000/documents/acknowledge/..."
   }
```

## 🧪 Testing Guide

### 1. Seed Default Templates

```bash
# Login as mediator
# Navigate to /mediator/documents
# Click "Create Default Templates"
# Creates 3 templates:
#   - Pre-Mediation Information Pack
#   - Fee Disclosure Statement  
#   - Mediation Agreement
```

### 2. Create Document from Template

```bash
# Navigate to case
# Click "Documents" tab
# Click "Add from Template"
# Select "Pre-Mediation Information Pack"
# System auto-fills variables
# Document created in "Draft" state
```

### 3. Send Document

```bash
# Click "Send" on document
# Enter email: test@example.com
# Click "Send"
# Check terminal for email log
# Document marked as "Sent"
# Copy acknowledgement URL from logs
```

### 4. Acknowledge Document

```bash
# Open acknowledgement URL in incognito window
# Should see public page (no login)
# Enter name: "Test Client"
# Click "I Acknowledge"
# See confirmation message
# Try accessing URL again → shows "Already acknowledged"
```

### 5. Verify in Database

```bash
npm run db:studio

# Check DocumentInstance:
# - sentAt: has timestamp
# - sentTo: "test@example.com"
# - acknowledgedAt: has timestamp
# - acknowledgedBy: "Test Client"
# - acknowledgementToken: (UUID)

# Check CaseTimeline:
# - Event: "Document Acknowledged"
# - Description: "Pre-Mediation Information Pack acknowledged by Test Client"
```

## 🎨 UI Features

### Template Management Page

```
Templates
├── Pre-Mediation Information Pack ✓
│   Category: Agreement
│   Requires: Acknowledgement
│   [Edit] [Preview] [Delete]
├── Fee Disclosure Statement ✓
│   Category: Invoice
│   Requires: Acknowledgement
│   [Edit] [Preview] [Delete]
└── [Create New Template]
```

### Case Documents Page

```
Documents for Case M8-2024-0042

[Add from Template ▼] [Upload File]

Draft Documents:
├── Mediation Agreement
│   Created: Jan 30, 2026
│   [Edit] [Send] [Delete]

Sent Documents:
├── Pre-Mediation Information Pack ✓
│   Sent: Jan 30, 2026 2:30 PM
│   To: jane@example.com
│   Acknowledged: Jan 30, 2026 3:15 PM by Jane Doe
│   [View] [Resend]

└── Fee Disclosure Statement
    Sent: Jan 30, 2026 2:35 PM
    To: jane@example.com, john@example.com
    Pending acknowledgement
    [View] [Resend]
```

### Acknowledgement Page (Public)

```
Document Acknowledgement

Case: M8-2024-0042
Document: Pre-Mediation Information Pack

[Document preview/content here]

By acknowledging, you confirm you have read and understood this document.

Your Name: [_______________]

[I Acknowledge]

This link expires on: Feb 29, 2026
```

## 📊 Template Variable System

### Available Variables

```typescript
// Case variables
{{caseNumber}}         // "M8-2024-0042"
{{caseDate}}           // "January 30, 2026"
{{disputeDescription}} // Case description

// Mediator variables
{{mediatorName}}       // "John Smith"
{{mediatorEmail}}      // "john@example.com"
{{mediatorPhone}}      // "+61 4XX XXX XXX"
{{yearsExperience}}    // "15"

// Pricing variables
{{hourlyRate}}         // "$200"
{{hourlyRateBasis}}    // "per party" or "total"
{{preMediationFee}}    // "$150"
{{depositAmount}}      // "$250"
{{travelIncludedKm}}   // "50"
{{travelRatePerKm}}    // "$1.50"

// Party variables
{{partyNames}}         // "Jane Doe, John Smith"
{{party1Name}}         // "Jane Doe"
{{party2Name}}         // "John Smith"

// System variables
{{currentDate}}        // "January 30, 2026"
{{appUrl}}             // "https://medi8.com.au"
```

### Conditional Blocks

```handlebars
{{#if preMediationFee}}
Pre-mediation fee: ${{preMediationFee}}
{{else}}
No pre-mediation fee applies
{{/if}}
```

## 🔄 Document Lifecycle

```
DRAFT
  ↓ (Send action)
SENT
  ↓ (Client acknowledges)
ACKNOWLEDGED
  ↓ (Client signs - future)
SIGNED
```

States:
- **DRAFT**: Created but not sent
- **SENT**: Emailed to client(s), awaiting acknowledgement
- **ACKNOWLEDGED**: Client confirmed receipt
- **SIGNED**: Client provided signature (Milestone 6)

## 📈 Statistics Tracked

### Per Mediator
- Total documents created
- Documents sent
- Documents acknowledged
- Pending acknowledgements
- Acknowledgement rate

### Per Case
- Documents attached
- Documents sent
- Documents acknowledged
- Outstanding documents

## 🐛 Error Handling

**Send Errors:**
- ✗ Document already sent → "Document has already been sent"
- ✗ No recipients → "At least one recipient required"
- ✗ Invalid email → "Invalid email address format"

**Acknowledge Errors:**
- ✗ Invalid token → "Document not found or invalid link"
- ✗ Expired token → "This acknowledgement link has expired"
- ✗ Already acknowledged → "This document has already been acknowledged"
- ✗ Empty name → "Please enter your name"

**Template Errors:**
- ✗ Template in use → "Cannot delete: template is used in X cases"
- ✗ Missing required field → "Name is required"
- ✗ Duplicate name → "A template with this name already exists"

## 🚀 Production Deployment

### Environment Variables

```bash
# Storage
STORAGE_PROVIDER=s3
AWS_S3_BUCKET=medi8-documents-prod
AWS_REGION=ap-southeast-2
AWS_ACCESS_KEY_ID=AKIAXXXXXXXX
AWS_SECRET_ACCESS_KEY=xxxx

# App
APP_URL=https://medi8.com.au
```

### Email Setup (Milestone 6)

Update `src/lib/email.ts` to use SendGrid/AWS SES:

```typescript
const provider = new SendGridEmailProvider(apiKey);
await provider.send({
  to: 'client@example.com',
  subject: 'Document sent',
  html: renderedTemplate,
});
```

### S3 Setup

```typescript
import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';

export class S3StorageProvider implements StorageProvider {
  async upload(file: File, path: string) {
    const s3 = new S3Client({ region: this.region });
    const buffer = Buffer.from(await file.arrayBuffer());
    
    await s3.send(new PutObjectCommand({
      Bucket: this.bucket,
      Key: path,
      Body: buffer,
      ContentType: file.type,
    }));
    
    return {
      path,
      url: `https://${this.bucket}.s3.${this.region}.amazonaws.com/${path}`,
      size: buffer.length,
      mimeType: file.type,
    };
  }
}
```

## ➡️ What's Next (Milestone 6)

Future enhancements:
- Document signing (e-signature)
- PDF generation from templates
- File uploads (attach existing files)
- Document versioning
- SendGrid/AWS SES integration
- S3 storage implementation
- Document search/filter
- Bulk send

---

**Status: ✅ Milestone 5 Complete**

Complete document management system with templates, per-case instances, send/acknowledge workflow, and storage abstraction!
