# Milestone 7: HubSpot Integration (Optional) - Complete!

## ✅ What's Implemented

### **HubSpot CRM Integration**
✅ **Contact Sync** - Auto-create/update contacts on enquiry  
✅ **Deal Creation** - Create deals for each case  
✅ **Deal Stage Updates** - Auto-update based on payment events  
✅ **Amount Tracking** - Sync payment amounts to deals  
✅ **Contact Association** - Link deals to contacts  
✅ **Activity Notes** - Add notes for key events  
✅ **Non-Blocking** - Failures don't break core system  
✅ **Toggleable** - Enable/disable via environment variable  

### **Automated Sync Events**
✅ **Case Created** → Contact + Deal (NEW_ENQUIRY stage)  
✅ **Deposit Paid** → Deal stage: DEPOSIT_PAID  
✅ **Approved to Proceed** → Deal stage: APPROVED_TO_PROCEED  
✅ **Balance Paid** → Deal stage: BALANCE_PAID + amount update  
✅ **Case Completed** → Deal stage: COMPLETED (closedwon)  
✅ **Case Cancelled** → Deal stage: CANCELLED (closedlost)  

### **Admin Features**
✅ **Integration Status** - View enabled/disabled state  
✅ **Connection Test** - Test API key validity  
✅ **Manual Sync** - Trigger sync for specific cases  
✅ **Setup Instructions** - Built-in docs  

## 🔄 Sync Flow

### Flow 1: New Enquiry Created

```
Client creates case
  ↓
medi8 creates Case record
  ↓
HubSpot Integration triggered
  ↓
Create/Update Contact:
  - Email: client@example.com
  - Name: Jane Doe
  - Phone: +61 4XX XXX XXX
  - Lifecycle: lead
  ↓
Create Deal:
  - Name: "M8-2024-0042 - Jane Doe"
  - Stage: appointmentscheduled (NEW_ENQUIRY)
  - Amount: $2,500 (quoted)
  ↓
Associate Deal with Contact
  ↓
Store HubSpot IDs in Case.metadata:
  {
    "hubspotDealId": "12345678",
    "hubspotContactId": "87654321",
    "hubspotSyncedAt": "2026-01-30T14:30:00Z"
  }
  ↓
Complete ✓ (non-blocking)
```

### Flow 2: Deposit Paid

```
Client pays $250 deposit
  ↓
Stripe webhook: payment_intent.succeeded
  ↓
Update payment status
  ↓
Update case: depositPaid = true
  ↓
HubSpot Integration triggered
  ↓
Update Deal Stage:
  - Stage: qualifiedtobuy (DEPOSIT_PAID)
  - Amount: $250
  ↓
Complete ✓
```

### Flow 3: Balance Paid

```
Client pays $2,250 balance
  ↓
Stripe webhook: payment_intent.succeeded
  ↓
Update payment status
  ↓
HubSpot Integration triggered
  ↓
Calculate total paid: $2,500
  ↓
Update Deal:
  - Stage: decisionmakerboughtin (BALANCE_PAID)
  - Amount: $2,500 (total)
  ↓
Complete ✓
```

### Flow 4: Case Completed

```
Mediator marks case complete
  ↓
HubSpot Integration triggered
  ↓
Calculate total revenue
  ↓
Update Deal:
  - Stage: closedwon (COMPLETED)
  - Amount: $2,500 (final)
  - Close Date: 2026-01-30
  ↓
Complete ✓
```

## 📊 HubSpot Deal Stages

Mapped to standard HubSpot stages:

| medi8 Event | HubSpot Stage | Stage ID |
|-------------|---------------|----------|
| Enquiry Created | Appointment Scheduled | `appointmentscheduled` |
| Deposit Paid | Qualified to Buy | `qualifiedtobuy` |
| Approved to Proceed | Presentation Scheduled | `presentationscheduled` |
| Balance Paid | Decision Maker Bought-In | `decisionmakerboughtin` |
| Case Completed | Closed Won | `closedwon` |
| Case Cancelled | Closed Lost | `closedlost` |

**Note:** Stage IDs can be customized in `/src/lib/hubspot.ts` to match your HubSpot pipeline.

## 🔐 Configuration

### Environment Variables

```bash
# .env.local
HUBSPOT_ENABLED=true
HUBSPOT_API_KEY=pat-na1-12345678-90ab-cdef-1234-567890abcdef
```

### Getting Your HubSpot API Key

1. Login to HubSpot
2. Settings → Integrations → **Private Apps**
3. Create Private App or select existing
4. **Required Scopes:**
   - `crm.objects.contacts.read`
   - `crm.objects.contacts.write`
   - `crm.objects.deals.read`
   - `crm.objects.deals.write`
   - `crm.schemas.contacts.read`
   - `crm.schemas.deals.read`
5. Generate API token
6. Copy token to `.env.local`

### Toggle Integration

```bash
# Enable
HUBSPOT_ENABLED=true

# Disable (default)
HUBSPOT_ENABLED=false
# or omit the variable entirely
```

## 📁 Files Created (6 files)

### Core Library
- `src/lib/hubspot.ts` - HubSpot client with API methods (370 lines)

### Service Layer
- `src/modules/integration/hubspot/service.ts` - Integration service (280 lines)
- `src/modules/integration/hubspot/actions.ts` - Server Actions (80 lines)

### Admin UI
- `src/app/(admin)/integrations/page.tsx` - Integration management page
- `src/components/integration/HubSpotIntegrationCard.tsx` - Status card

### Webhook Updates
- `src/app/api/webhooks/stripe/route.ts` - Added HubSpot sync calls

### Database
- `prisma/schema.prisma` - Added `metadata` JSON field to Case model

## 🔧 Technical Implementation

### Non-Blocking Architecture

```typescript
// Failures don't break core system
try {
  const hubspotService = new HubSpotIntegrationService();
  await hubspotService.syncDepositPaid(caseId);
} catch (error) {
  console.error('[HubSpot] Sync failed (non-blocking):', error);
  // Core flow continues regardless
}
```

### Graceful Degradation

```typescript
// Check if enabled before every operation
if (!this.isEnabled()) {
  console.log('[HubSpot] Skipped - integration disabled');
  return null;
}

// All API calls wrapped in try/catch
try {
  await this.client.post('/crm/v3/objects/contacts', data);
} catch (error) {
  console.error('[HubSpot] Failed:', error);
  return null; // Return null, don't throw
}
```

### Metadata Storage

```typescript
// Store HubSpot IDs in Case metadata
await prisma.case.update({
  where: { id: caseId },
  data: {
    metadata: {
      hubspotDealId: "12345678",
      hubspotContactId: "87654321",
      hubspotSyncedAt: "2026-01-30T14:30:00Z"
    }
  }
});

// Retrieve for updates
const metadata = caseRecord.metadata as any;
const dealId = metadata.hubspotDealId;
```

## 🧪 Testing

### Test 1: Connection Test

```bash
# 1. Set environment variables
HUBSPOT_ENABLED=true
HUBSPOT_API_KEY=pat-na1-...

# 2. Restart app
npm run dev

# 3. Login as admin
# 4. Navigate to /admin/integrations
# 5. Click "Test Connection"
# 6. Should see: "Successfully connected to HubSpot"
```

### Test 2: Case Creation Sync

```bash
# 1. Create a new case as client
# 2. Check console logs:
#    [HubSpot] Created contact: 87654321
#    [HubSpot] Created deal: 12345678
#    [HubSpot] Associated deal with contact
#    [HubSpot] Synced case M8-2024-0042
# 3. Check HubSpot:
#    - Contact exists with correct email
#    - Deal exists with case number in name
#    - Deal associated with contact
#    - Deal in "Appointment Scheduled" stage
```

### Test 3: Deposit Payment Sync

```bash
# 1. Pay deposit for case
# 2. Stripe webhook fires
# 3. Check console logs:
#    [HubSpot] Updated deal 12345678 to DEPOSIT_PAID
# 4. Check HubSpot:
#    - Deal moved to "Qualified to Buy" stage
#    - Deal amount updated to $250
```

### Test 4: Balance Payment Sync

```bash
# 1. Pay balance ($2,250)
# 2. Check console logs:
#    [HubSpot] Updated deal 12345678 to BALANCE_PAID with amount: $2500
# 3. Check HubSpot:
#    - Deal moved to "Decision Maker Bought-In"
#    - Deal amount updated to $2,500 (total)
```

### Test 5: Disabled Integration

```bash
# 1. Set HUBSPOT_ENABLED=false
# 2. Restart app
# 3. Create case
# 4. Check console logs:
#    [HubSpot] Skipped - integration disabled
# 5. Verify core system works normally
# 6. No errors or crashes
```

## 🔍 Admin Dashboard

### Integration Status Card

```
┌─────────────────────────────────────────┐
│ 🟠 HubSpot CRM              [Active]   │
│ Sync contacts and deals automatically   │
│                                          │
│ Integration Enabled     ✓ Yes           │
│ API Key Configured      ✓ Yes           │
│                                          │
│ Features:                                │
│ ✓ Auto-create contacts on enquiry       │
│ ✓ Create deals for each case            │
│ ✓ Update stages on deposit/balance paid │
│ ✓ Sync deal amounts automatically       │
│ ✓ Non-blocking (failures don't break)   │
│                                          │
│ [Test Connection]                        │
│                                          │
│ ✓ Successfully connected to HubSpot     │
└─────────────────────────────────────────┘
```

## 📊 HubSpot Contact Properties

Created/updated on each contact:

```typescript
{
  email: "client@example.com",    // Required
  firstname: "Jane",               // From client record
  lastname: "Doe",                 // From client record
  phone: "+61 4XX XXX XXX",        // From client record
  lifecyclestage: "lead"           // Set on creation
}
```

## 📊 HubSpot Deal Properties

Created/updated on each deal:

```typescript
{
  dealname: "M8-2024-0042 - Jane Doe",
  amount: 2500,                    // In dollars
  dealstage: "appointmentscheduled",
  pipeline: "default",             // Or custom
  closedate: "2026-01-30T00:00:00Z"
}
```

## 🛡️ Error Handling

### Scenarios Handled

✅ **No API Key** - Integration silently disabled  
✅ **Invalid API Key** - Connection test fails, syncs skip  
✅ **Rate Limiting** - Errors logged, doesn't crash  
✅ **Network Timeout** - 10-second timeout, then fail gracefully  
✅ **Contact Already Exists** - Update instead of create  
✅ **Deal Not Found** - Log error, continue  
✅ **Missing Metadata** - Skip sync, no error  

### Error Logging

```typescript
console.error('[HubSpot] Failed to create contact:', error);
// Errors prefixed with [HubSpot] for easy filtering
// Non-blocking - core system unaffected
```

## 🎯 Use Cases

### Use Case 1: Sales Pipeline Tracking

**Goal:** Track mediation cases through sales pipeline

**Setup:**
1. Enable HubSpot integration
2. Cases auto-create as deals
3. Stages update automatically
4. Sales team sees real-time progress

**Benefits:**
- No manual data entry
- Accurate pipeline view
- Revenue forecasting
- Conversion tracking

### Use Case 2: Client Relationship Management

**Goal:** Maintain complete client history

**Setup:**
1. Contacts auto-created on first enquiry
2. Multiple cases linked to same contact
3. Activity timeline auto-populated

**Benefits:**
- Complete client history
- Multi-case tracking
- Relationship building
- Retention insights

### Use Case 3: Marketing Attribution

**Goal:** Track which marketing efforts generate cases

**Setup:**
1. HubSpot tracks contact source
2. Deals created with case info
3. Revenue attributed to source

**Benefits:**
- ROI measurement
- Channel optimization
- Budget allocation
- Campaign tracking

## 🔄 Sync Triggers

| Event | Trigger Point | HubSpot Action |
|-------|---------------|----------------|
| Case Created | After Case insert | Create Contact + Deal |
| Deposit Paid | Stripe webhook success | Update Deal stage |
| Approved | Admin approval | Update Deal stage |
| Balance Paid | Stripe webhook success | Update Deal stage + amount |
| Completed | Case status change | Update Deal stage + close date |
| Cancelled | Case status change | Update Deal stage + note |

## 📈 Future Enhancements

Potential additions (not in this milestone):

- ✨ Bidirectional sync (HubSpot → medi8)
- ✨ Custom property mapping
- ✨ Multiple pipeline support
- ✨ Task creation for follow-ups
- ✨ Email integration
- ✨ Meeting scheduling
- ✨ Advanced reporting
- ✨ Workflow automation

## 🐛 Troubleshooting

### Issue: "Integration disabled"

**Solution:**
```bash
# Check .env.local
HUBSPOT_ENABLED=true  # Must be exactly "true"
HUBSPOT_API_KEY=pat-...  # Must be set

# Restart app
npm run dev
```

### Issue: "Connection test failed"

**Causes:**
- Invalid API key
- Wrong scopes
- Expired token
- Network issues

**Solution:**
1. Verify API key is correct
2. Check scopes in HubSpot
3. Regenerate token if needed
4. Check network/firewall

### Issue: "Contact not syncing"

**Check:**
1. Is integration enabled?
2. Check console for errors
3. Verify case has primary party
4. Check party has email

### Issue: "Deal stage not updating"

**Check:**
1. Does case have metadata with dealId?
2. Check console logs for errors
3. Verify stage ID exists in HubSpot
4. Check API key permissions

## 🚀 Production Deployment

### Environment Variables

```bash
# Production .env
HUBSPOT_ENABLED=true
HUBSPOT_API_KEY=pat-na1-production-key-here
```

### Deployment Checklist

- [ ] Set HUBSPOT_ENABLED=true
- [ ] Add production API key
- [ ] Test connection in admin panel
- [ ] Create test case to verify sync
- [ ] Monitor logs for errors
- [ ] Check HubSpot for test data
- [ ] Clear test data if needed

### Monitoring

```bash
# Check logs for HubSpot activity
grep "\[HubSpot\]" logs.txt

# Look for errors
grep "\[HubSpot\].*Failed" logs.txt

# Verify successful syncs
grep "\[HubSpot\].*Synced case" logs.txt
```

## 💡 Best Practices

### Do's ✅

- Test connection before going live
- Monitor logs regularly
- Keep API key secure
- Use descriptive deal names
- Map to standard stages when possible

### Don'ts ❌

- Don't hardcode API key in code
- Don't throw errors on sync failures
- Don't block core operations on HubSpot
- Don't sync PII without consent
- Don't spam HubSpot API (rate limits)

## 📚 API Reference

### HubSpotClient Methods

```typescript
// Create/update contact
await hubspot.upsertContact({ email, firstname, lastname });

// Create deal
await hubspot.createDeal({ dealname, amount, dealstage });

// Update deal stage
await hubspot.updateDealStage(dealId, stage);

// Update deal properties
await hubspot.updateDeal(dealId, { amount: 2500 });

// Associate deal with contact
await hubspot.associateDealWithContact(dealId, contactId);

// Add note to contact
await hubspot.addNoteToContact(contactId, "Note text");

// Test connection
await hubspot.testConnection();
```

### HubSpotIntegrationService Methods

```typescript
// Sync case created
await service.syncCaseCreated(caseId);

// Sync deposit paid
await service.syncDepositPaid(caseId);

// Sync approved to proceed
await service.syncApprovedToProceed(caseId);

// Sync balance paid
await service.syncBalancePaid(caseId, totalAmount);

// Sync case completed
await service.syncCaseCompleted(caseId);

// Sync case cancelled
await service.syncCaseCancelled(caseId, reason);

// Add activity note
await service.addActivityNote(caseId, "Activity text");
```

---

**Status: ✅ Milestone 7 Complete (Optional)**

Complete HubSpot CRM integration with automatic contact/deal sync, stage updates on payment events, and graceful failure handling. Fully toggleable and non-blocking!
