# medi8 - Mediator Practice OS

A production-ready, multi-tenant SaaS platform for mediators to manage their practice, cases, payments, and client relationships.

## Features (Milestone 1: Foundation)

✅ **Authentication & Authorization**
- Secure session-based authentication with httpOnly cookies
- Role-based access control (CLIENT, MEDIATOR, ADMIN)
- Password hashing with bcrypt
- Session expiration and cleanup

✅ **Multi-Tenant Architecture**
- Strict data isolation between mediators
- Role-based route protection
- Audit logging for all user actions

✅ **Database Schema**
- Complete Prisma schema with all entities
- Support for GST registration tracking
- Subscription management (trial, active, past due)
- Payment fee calculation (Stripe fees + platform fees + GST)

✅ **Admin Bootstrap**
- Automatic admin user creation on first seed
- Platform configuration management
- Notification templates

## Tech Stack

- **Framework**: Next.js 14 (App Router)
- **Language**: TypeScript
- **Database**: PostgreSQL
- **ORM**: Prisma
- **Authentication**: Custom (session-based)
- **Styling**: Tailwind CSS
- **Payments**: Stripe Connect (ready for Milestone 5)

## Prerequisites

- Node.js 18+ and npm/yarn
- PostgreSQL 14+
- Git

## Getting Started

### 1. Clone the repository

\`\`\`bash
git clone <repository-url>
cd medi8
\`\`\`

### 2. Install dependencies

\`\`\`bash
npm install
\`\`\`

### 3. Set up environment variables

Copy the example environment file:

\`\`\`bash
cp .env.example .env.local
\`\`\`

Edit \`.env.local\` and configure:

\`\`\`bash
# Database - Update with your PostgreSQL credentials
DATABASE_URL="postgresql://postgres:password@localhost:5432/medi8?schema=public"

# App URL
APP_URL="http://localhost:3000"
NODE_ENV="development"

# Stripe (for Milestone 5 - can leave as-is for now)
STRIPE_SECRET_KEY="sk_test_..."
STRIPE_PUBLISHABLE_KEY="pk_test_..."
STRIPE_WEBHOOK_SECRET="whsec_..."

# Storage (local for development)
STORAGE_PROVIDER="local"

# Email (console for development - logs to terminal)
EMAIL_PROVIDER="console"
\`\`\`

### 4. Set up the database

#### Option A: Create PostgreSQL database manually

\`\`\`bash
# Connect to PostgreSQL
psql -U postgres

# Create database
CREATE DATABASE medi8;

# Exit psql
\q
\`\`\`

#### Option B: Use Docker for PostgreSQL

\`\`\`bash
docker run --name medi8-postgres \\
  -e POSTGRES_PASSWORD=password \\
  -e POSTGRES_DB=medi8 \\
  -p 5432:5432 \\
  -d postgres:14
\`\`\`

### 5. Run database migrations

\`\`\`bash
# Generate Prisma client
npm run db:generate

# Run migrations
npm run db:migrate

# Seed initial data (creates admin user, platform config, email templates)
npm run db:seed
\`\`\`

**Important**: The seed script creates a default admin user:
- Email: `admin@medi8.com.au`
- Password: `admin123`

⚠️ **Change this password immediately after first login!**

### 6. Start the development server

\`\`\`bash
npm run dev
\`\`\`

Visit [http://localhost:3000](http://localhost:3000)

## Default Login Credentials

After running the seed script:

**Admin User:**
- Email: `admin@medi8.com.au`
- Password: `admin123`

**Test Mediator** (create via registration):
- Visit: http://localhost:3000/register/mediator
- Register with any email
- GST registration is optional

**Test Client** (create via registration):
- Visit: http://localhost:3000/register/client
- Register with any email

## Project Structure

\`\`\`
medi8/
├── prisma/
│   ├── schema.prisma          # Database schema
│   ├── seed.ts                # Seed script
│   └── migrations/            # Migration files
├── src/
│   ├── app/                   # Next.js App Router
│   │   ├── (admin)/          # Admin dashboard (protected)
│   │   ├── (mediator)/       # Mediator dashboard (protected)
│   │   ├── (client)/         # Client dashboard (protected)
│   │   ├── login/            # Login page
│   │   ├── register/         # Registration pages
│   │   ├── layout.tsx        # Root layout
│   │   └── page.tsx          # Homepage
│   ├── components/           # Reusable components
│   │   └── DashboardNav.tsx  # Navigation component
│   ├── lib/                  # Utilities
│   │   ├── db.ts             # Prisma client singleton
│   │   ├── email.ts          # Email abstraction
│   │   └── storage.ts        # File storage abstraction
│   ├── modules/              # Business logic modules
│   │   ├── auth/             # Authentication module
│   │   │   ├── actions.ts    # Server Actions
│   │   │   ├── service.ts    # Business logic
│   │   │   ├── validation.ts # Zod schemas
│   │   │   └── types.ts      # TypeScript types
│   │   └── shared/           # Shared utilities
│   │       ├── audit.ts      # Audit logging
│   │       └── errors/       # Custom error classes
│   └── middleware.ts         # Route protection middleware
├── .env.example              # Environment variables template
├── .gitignore
├── package.json
├── tsconfig.json
└── README.md
\`\`\`

## Database Commands

\`\`\`bash
# Generate Prisma Client (after schema changes)
npm run db:generate

# Create a new migration
npm run db:migrate

# Apply migrations in production
npm run db:migrate:prod

# Open Prisma Studio (database GUI)
npm run db:studio

# Re-seed database (WARNING: deletes all data)
npm run db:reset

# Seed database only
npm run db:seed
\`\`\`

## User Flows (Milestone 1)

### Admin Flow
1. Login with default credentials
2. Access admin dashboard at `/admin/dashboard`
3. View platform statistics
4. (Approvals and audit log in Milestone 2)

### Mediator Flow
1. Register at `/register/mediator`
2. Provide GST information (optional)
3. Auto-enrolled in 30-day trial
4. Status: PENDING_APPROVAL
5. Access dashboard at `/mediator/dashboard`
6. View trial days remaining
7. (Profile editing in Milestone 2)

### Client Flow
1. Register at `/register/client`
2. Access dashboard at `/client/dashboard`
3. (Find mediators in Milestone 3)
4. (Submit inquiries in Milestone 4)

## Middleware & Route Protection

The app uses Next.js middleware for authentication and role-based access:

- **Public routes**: `/`, `/login`, `/register/*`, `/mediators/*`
- **Admin routes**: `/admin/*` (requires ADMIN role)
- **Mediator routes**: `/mediator/*` (requires MEDIATOR role)
- **Client routes**: `/client/*` (requires CLIENT role)

Unauthorized access redirects to `/login`.

## Email Notifications

Emails are abstracted via the \`EmailProvider\` interface:

**Development**: Uses \`ConsoleEmailProvider\` (logs to terminal)
**Production**: Ready for \`SendGridEmailProvider\` (implement in Milestone 6)

Email templates are stored in the database and support placeholders like \`{{firstName}}\`.

## File Storage

File storage is abstracted via the \`StorageProvider\` interface:

**Development**: Uses \`LocalStorageProvider\` (saves to \`public/uploads\`)
**Production**: Ready for \`S3StorageProvider\` (implement in Milestone 6)

## Testing Authentication

### Test Admin Access
\`\`\`bash
# Login as admin
Email: admin@medi8.com.au
Password: admin123

# Should redirect to /admin/dashboard
\`\`\`

### Test Mediator Registration
1. Visit `/register/mediator`
2. Fill in details
3. Check "I am registered for GST" (optional)
4. Enter GST number if applicable
5. Submit form
6. Should redirect to `/mediator/dashboard`
7. Should see trial status banner

### Test Client Registration
1. Visit `/register/client`
2. Fill in details
3. Submit form
4. Should redirect to `/client/dashboard`

## Audit Logging

All important actions are logged to the \`AuditLog\` table:

- User registration
- Login/logout
- Profile changes
- Admin actions

View logs in Prisma Studio:
\`\`\`bash
npm run db:studio
# Navigate to AuditLog table
\`\`\`

## Platform Configuration

Platform settings are stored in the \`PlatformConfig\` table:

| Key | Default Value | Description |
|-----|---------------|-------------|
| PLATFORM_FEE_PERCENT | 0.10 | Platform fee (10% of net) |
| SUBSCRIPTION_MONTHLY_AMOUNT | 50.00 | Monthly subscription (excl GST) |
| SUBSCRIPTION_ANNUAL_AMOUNT | 600.00 | Annual subscription (excl GST) |
| GST_RATE | 0.10 | GST rate (10%) |
| TRIAL_PERIOD_DAYS | 30 | Trial period in days |

## Next Steps (Upcoming Milestones)

### Milestone 2: Mediator Onboarding & Admin Approval
- Profile editing (bio, certifications, specializations)
- Pricing configuration
- Photo upload
- Admin approval workflow
- Trial expiry management

### Milestone 3: Public Discovery & Practice Pages
- Public mediator directory
- Hosted practice pages (\`/mediators/[slug]\`)
- Search and filtering

### Milestone 4: Case Management & Client Portal
- Inquiry submission
- Case creation and workflow
- Multi-party support
- Timeline tracking

### Milestone 5: Payments (Stripe Connect + Subscriptions)
- Stripe Connect onboarding
- Payment processing with correct fee structure
- Platform subscriptions (monthly/annual)
- GST handling
- Webhook processing

### Milestone 6: Documents, Notifications & Admin Oversight
- Document upload/download
- Email notifications (SendGrid)
- Audit log viewer
- Platform configuration UI

## Troubleshooting

### Database Connection Issues

**Error**: \`Can't reach database server\`
- Check PostgreSQL is running: \`pg_isready\`
- Verify \`DATABASE_URL\` in \`.env.local\`
- Check firewall/port 5432

### Migration Errors

**Error**: \`Migration already applied\`
- Reset database: \`npm run db:reset\` (WARNING: deletes data)
- Or create new migration: \`npm run db:migrate\`

### Prisma Client Errors

**Error**: \`PrismaClient is unable to be run in the browser\`
- Ensure you're not importing Prisma in client components
- Only use Prisma in Server Components or Server Actions

### Session/Cookie Issues

**Error**: Session not persisting
- Check cookie settings in browser devtools
- Ensure \`httpOnly\` cookies are enabled
- Try clearing browser cookies

## Environment Variables Reference

| Variable | Required | Default | Description |
|----------|----------|---------|-------------|
| DATABASE_URL | Yes | - | PostgreSQL connection string |
| APP_URL | Yes | http://localhost:3000 | Application URL |
| NODE_ENV | No | development | Environment (development/production) |
| STRIPE_SECRET_KEY | No* | - | Stripe secret key (*required in M5) |
| STRIPE_PUBLISHABLE_KEY | No* | - | Stripe public key (*required in M5) |
| STRIPE_WEBHOOK_SECRET | No* | - | Stripe webhook secret (*required in M5) |
| STORAGE_PROVIDER | No | local | Storage provider (local/s3) |
| EMAIL_PROVIDER | No | console | Email provider (console/sendgrid) |
| AWS_S3_BUCKET | No | - | S3 bucket name (if provider=s3) |
| AWS_REGION | No | - | AWS region (if provider=s3) |
| SENDGRID_API_KEY | No | - | SendGrid API key (if provider=sendgrid) |

## Contributing

1. Create a feature branch
2. Make your changes
3. Run tests (when available)
4. Submit a pull request

## License

Proprietary - All rights reserved

## Support

For issues or questions:
- Open a GitHub issue
- Contact: dev@medi8.com.au

---

**Current Status**: ✅ Milestone 1 Complete (Foundation & Authentication)

**Next**: Milestone 2 - Mediator Onboarding & Admin Approval
