# Stripe Webhook Testing Guide

This guide explains how to test Stripe webhooks locally during development.

## Prerequisites

- Stripe CLI installed
- medi8 application running locally
- Stripe account (test mode)

## Option 1: Stripe CLI (Recommended)

### 1. Install Stripe CLI

**macOS:**
```bash
brew install stripe/stripe-cli/stripe
```

**Linux:**
```bash
# Download and install from https://github.com/stripe/stripe-cli/releases
wget https://github.com/stripe/stripe-cli/releases/latest/download/stripe_linux_x86_64.tar.gz
tar -xvf stripe_linux_x86_64.tar.gz
sudo mv stripe /usr/local/bin/
```

**Windows:**
```bash
# Download from https://github.com/stripe/stripe-cli/releases
# Or use Scoop
scoop install stripe
```

### 2. Login to Stripe

```bash
stripe login
```

This will open a browser to authenticate with your Stripe account.

### 3. Forward Webhooks to Local Server

```bash
# Forward all webhook events to your local endpoint
stripe listen --forward-to localhost:3000/api/webhooks/stripe
```

**Output:**
```
> Ready! Your webhook signing secret is whsec_xxxxxxxxxxxxx (^C to quit)
```

### 4. Copy the Webhook Secret

Copy the `whsec_` secret from the terminal output and add it to `.env.local`:

```bash
STRIPE_WEBHOOK_SECRET="whsec_xxxxxxxxxxxxx"
```

### 5. Restart Your App

```bash
# Restart to pick up the new env variable
npm run dev
```

### 6. Keep Stripe CLI Running

Leave the `stripe listen` command running in a separate terminal window.

## Option 2: ngrok (Alternative)

If you need a public URL for testing:

### 1. Install ngrok

```bash
# macOS
brew install ngrok

# Or download from https://ngrok.com/download
```

### 2. Start ngrok

```bash
ngrok http 3000
```

**Output:**
```
Forwarding https://xxxx-xx-xx-xx-xx.ngrok.io -> http://localhost:3000
```

### 3. Add Webhook in Stripe Dashboard

1. Go to: https://dashboard.stripe.com/test/webhooks
2. Click **"Add endpoint"**
3. Endpoint URL: `https://xxxx-xx-xx-xx-xx.ngrok.io/api/webhooks/stripe`
4. Select events to send:
   - `payment_intent.succeeded`
   - `payment_intent.payment_failed`
   - `charge.refunded`
   - `account.updated`
5. Copy the **Signing secret** (starts with `whsec_`)
6. Add to `.env.local`:
   ```bash
   STRIPE_WEBHOOK_SECRET="whsec_xxxxxxxxxxxxx"
   ```

## Testing Payment Webhooks

### Test 1: Deposit Payment

```bash
# In one terminal, run the Stripe CLI listener
stripe listen --forward-to localhost:3000/api/webhooks/stripe

# In another terminal, trigger a test payment
stripe trigger payment_intent.succeeded
```

**What happens:**
1. Stripe CLI sends a `payment_intent.succeeded` event
2. Your webhook endpoint receives it
3. Payment status updates to SUCCEEDED
4. Case status updates to BOOKED (if deposit)
5. Email notification sent to mediator
6. Check logs in terminal

### Test 2: Failed Payment

```bash
stripe trigger payment_intent.payment_failed
```

**What happens:**
1. Payment status updates to FAILED
2. Failure reason captured
3. Client notified (if email template exists)

### Test 3: Refund

```bash
stripe trigger charge.refunded
```

**What happens:**
1. Payment status updates to REFUNDED
2. Refund amount recorded

### Test 4: Connect Account Update

```bash
stripe trigger account.updated
```

**What happens:**
1. Mediator onboarding status checked
2. `stripeOnboarded` flag updated if complete

## Manual Testing with Real Payments

### 1. Create a Test Payment (via UI)

1. Login as a client
2. Navigate to a case
3. Click "Pay Deposit" or "Pay Balance"
4. Use Stripe test card: `4242 4242 4242 4242`
5. Any future date, any CVC

### 2. Watch Webhook Logs

In the terminal running `stripe listen`:
```
2024-01-30 14:32:45  --> payment_intent.succeeded [evt_xxxxx]
2024-01-30 14:32:45  <-- [200] POST http://localhost:3000/api/webhooks/stripe [evt_xxxxx]
```

### 3. Check Database

```bash
# Open Prisma Studio
npm run db:studio

# Check:
# - Payment status = SUCCEEDED
# - Case depositPaid = true (if deposit)
# - WebhookEvent processed = true
```

## Webhook Event Flow

### Deposit Payment Flow

```
Client initiates deposit payment
  ↓
PaymentIntent created ($250, no destination)
  ↓
Client enters card details
  ↓
Payment processes
  ↓
Webhook: payment_intent.succeeded
  ↓
Update payment status → SUCCEEDED
  ↓
Update case depositPaid → true
  ↓
Update case status → BOOKED
  ↓
Send email to mediator
  ↓
Complete ✓
```

### Balance Payment Flow

```
Client initiates balance payment
  ↓
PaymentIntent created with destination charge
  ↓
- Amount: e.g., $1000
- Application fee: $109.70 (10% platform + GST)
- Destination: mediator's Stripe account
  ↓
Client enters card details
  ↓
Payment processes
  ↓
Webhook: payment_intent.succeeded
  ↓
Update payment status → SUCCEEDED
  ↓
Funds transferred:
- Stripe keeps: $29.30 (2.9% + $0.30)
- Platform keeps: $109.70 (fee + GST)
- Mediator receives: $860.93
  ↓
Send email to mediator
  ↓
Complete ✓
```

## Test Cards

### Successful Payments
```
4242 4242 4242 4242  - Succeeds immediately
4000 0027 6000 3184  - Requires authentication (3D Secure)
```

### Failed Payments
```
4000 0000 0000 0002  - Card declined
4000 0000 0000 9995  - Insufficient funds
4000 0000 0000 0069  - Card expired
```

### Refunds
```
# Pay with any successful card
# Then in Stripe Dashboard → Payments → Refund
```

## Debugging Webhooks

### Check Webhook Logs

```bash
# In the Stripe CLI terminal, you'll see:
stripe listen --forward-to localhost:3000/api/webhooks/stripe

# Output shows:
# - Event type
# - HTTP status code
# - Response time
# - Any errors
```

### Check Application Logs

```bash
# In your Next.js terminal, you'll see:
Payment succeeded: pi_xxxxx
Payment processed successfully: pay_xxxxx
```

### Check Database

```sql
-- Check webhook events
SELECT * FROM "WebhookEvent" ORDER BY "createdAt" DESC LIMIT 10;

-- Check if event was processed
SELECT "eventId", "processed", "processedAt", "errorMessage" 
FROM "WebhookEvent" 
WHERE "eventId" = 'evt_xxxxx';

-- Check payment status
SELECT * FROM "Payment" WHERE "stripePaymentIntentId" = 'pi_xxxxx';
```

### Common Issues

**Issue: Webhook signature verification failed**
```
Solution: Check STRIPE_WEBHOOK_SECRET in .env.local matches CLI output
```

**Issue: Event already processed**
```
Solution: Normal - idempotency working correctly
```

**Issue: Payment record not found**
```
Solution: 
1. Check payment was created before webhook
2. Check stripePaymentIntentId matches
3. Check metadata on PaymentIntent
```

**Issue: Webhook not firing**
```
Solution:
1. Ensure Stripe CLI is running
2. Check endpoint URL is correct
3. Verify Next.js app is running on port 3000
```

## Production Webhook Setup

### 1. Deploy Your App

```bash
# Deploy to production (e.g., Vercel)
vercel --prod
```

### 2. Add Webhook Endpoint in Stripe

1. Go to: https://dashboard.stripe.com/webhooks
2. Click **"Add endpoint"**
3. Endpoint URL: `https://yourdomain.com/api/webhooks/stripe`
4. Select events:
   - `payment_intent.succeeded`
   - `payment_intent.payment_failed`
   - `charge.refunded`
   - `account.updated`
5. Copy signing secret
6. Add to production environment variables

### 3. Test Production Webhooks

```bash
# Use Stripe CLI to forward events to production
stripe listen --forward-to https://yourdomain.com/api/webhooks/stripe
```

## Webhook Endpoint Details

### Endpoint
```
POST /api/webhooks/stripe
```

### Headers Required
```
stripe-signature: t=xxx,v1=yyy,v0=zzz
```

### Events Handled
- `payment_intent.succeeded` - Payment successful
- `payment_intent.payment_failed` - Payment failed
- `charge.refunded` - Payment refunded
- `account.updated` - Connect account updated

### Response
```json
{ "received": true }
```

### Idempotency
- Events stored in `WebhookEvent` table
- Duplicate events automatically ignored
- Retry count tracked for failures

## Monitoring

### Check Webhook Health

```bash
# Get recent webhook events
stripe events list --limit 10

# Get specific event
stripe events retrieve evt_xxxxx

# Retry failed webhook
stripe events resend evt_xxxxx
```

### Database Queries

```sql
-- Unprocessed webhooks
SELECT * FROM "WebhookEvent" 
WHERE "processed" = false 
ORDER BY "createdAt" DESC;

-- Failed webhooks
SELECT * FROM "WebhookEvent" 
WHERE "errorMessage" IS NOT NULL 
ORDER BY "createdAt" DESC;

-- Recent payments
SELECT * FROM "Payment" 
ORDER BY "createdAt" DESC 
LIMIT 10;
```

## Summary

✅ **For local development:** Use Stripe CLI  
✅ **For public testing:** Use ngrok  
✅ **For production:** Add webhook in Stripe Dashboard  
✅ **Always:** Keep webhook secret in environment variables  
✅ **Monitor:** Check logs and database for issues  

---

**Quick Start:**
```bash
# Terminal 1: Start app
npm run dev

# Terminal 2: Start Stripe CLI
stripe listen --forward-to localhost:3000/api/webhooks/stripe

# Copy webhook secret to .env.local
# Restart app
# Test with: stripe trigger payment_intent.succeeded
```
