import { PrismaClient } from '@prisma/client';
import bcrypt from 'bcrypt';

const prisma = new PrismaClient();

async function main() {
  console.log('🌱 Starting seed...');

  // Check if admin exists
  const existingAdmin = await prisma.user.findFirst({
    where: { role: 'ADMIN' },
  });

  if (existingAdmin) {
    console.log('✅ Admin user already exists');
  } else {
    // Create admin user
    const adminPassword = await bcrypt.hash('admin123', 10);
    const admin = await prisma.user.create({
      data: {
        email: 'admin@medi8.com.au',
        passwordHash: adminPassword,
        role: 'ADMIN',
      },
    });
    console.log('✅ Created admin user:', admin.email);
    console.log('   Password: admin123');
  }

  // Platform configuration
  const configs = [
    {
      key: 'PLATFORM_FEE_PERCENT',
      value: '0.10',
      description: 'Platform fee percentage (10% of net after Stripe fees)',
    },
    {
      key: 'SUBSCRIPTION_MONTHLY_AMOUNT',
      value: '50.00',
      description: 'Monthly subscription fee (excluding GST)',
    },
    {
      key: 'SUBSCRIPTION_ANNUAL_AMOUNT',
      value: '600.00',
      description: 'Annual subscription fee (excluding GST)',
    },
    {
      key: 'GST_RATE',
      value: '0.10',
      description: 'GST rate (10%)',
    },
    {
      key: 'TRIAL_PERIOD_DAYS',
      value: '30',
      description: 'Trial period in days',
    },
  ];

  for (const config of configs) {
    await prisma.platformConfig.upsert({
      where: { key: config.key },
      update: {},
      create: config,
    });
  }
  console.log('✅ Created platform configuration');

  // Notification templates
  const templates = [
    {
      key: 'MEDIATOR_APPROVED',
      subject: 'Your medi8 profile has been approved!',
      bodyTemplate: `Hi {{firstName}},

Great news! Your mediator profile has been approved by our team.

You can now publish your profile and start receiving client inquiries.

View your profile: {{appUrl}}/mediators/{{slug}}

Best regards,
The medi8 Team`,
    },
    {
      key: 'MEDIATOR_REJECTED',
      subject: 'Update on your medi8 mediator application',
      bodyTemplate: `Hi {{firstName}},

Thank you for your interest in joining medi8.

Unfortunately, we are unable to approve your mediator profile at this time.

Reason: {{reason}}

If you have questions, please contact our support team.

Best regards,
The medi8 Team`,
    },
    {
      key: 'SUBSCRIPTION_PAYMENT_FAILED',
      subject: 'Payment failed for your medi8 subscription',
      bodyTemplate: `Hi {{firstName}},

We were unable to process your subscription payment.

Please update your payment method to continue using medi8.

Update payment: {{appUrl}}/mediator/payments/subscription

Best regards,
The medi8 Team`,
    },
    {
      key: 'TRIAL_EXPIRED',
      subject: 'Your medi8 trial has expired',
      bodyTemplate: `Hi {{firstName}},

Your 30-day trial period has ended.

To continue using medi8, please subscribe to one of our plans:
- Monthly: $55/month (inc GST)
- Annual: $660/year (inc GST)

Subscribe now: {{appUrl}}/mediator/payments/subscription

Best regards,
The medi8 Team`,
    },
    {
      key: 'PAYMENT_RECEIVED',
      subject: 'Payment received for case {{caseNumber}}',
      bodyTemplate: `Hi {{firstName}},

A payment of ${{amount}} has been received for case {{caseNumber}}.

The funds will be transferred to your account according to your payout schedule.

View case: {{appUrl}}/mediator/cases/{{caseId}}

Best regards,
The medi8 Team`,
    },
    {
      key: 'CASE_BOOKED',
      subject: 'New case booked: {{caseNumber}}',
      bodyTemplate: `Hi {{firstName}},

A new case has been booked!

Case: {{caseNumber}}
Client: {{clientName}}
Deposit: ${{depositAmount}}

View case: {{appUrl}}/mediator/cases/{{caseId}}

Best regards,
The medi8 Team`,
    },
    {
      key: 'MEDIATOR_PUBLISHED',
      subject: 'Your practice page is now live!',
      bodyTemplate: `Hi {{firstName}},

Congratulations! Your mediator profile is now published and live on medi8.

Your practice page: {{practiceUrl}}

Clients can now find and contact you through the platform.

Best regards,
The medi8 Team`,
    },
    {
      key: 'MEDIATOR_SUSPENDED',
      subject: 'Your medi8 account has been suspended',
      bodyTemplate: `Hi {{firstName}},

Your mediator account has been suspended.

Reason: {{reason}}

If you believe this is an error, please contact our support team.

Best regards,
The medi8 Team`,
    },
    {
      key: 'DOCUMENT_SENT',
      subject: 'Document: {{documentName}} for case {{caseNumber}}',
      bodyTemplate: `Hi {{firstName}},

You have received a new document for case {{caseNumber}}.

Document: {{documentName}}

{{message}}

{{#if requiresAcknowledgement}}
This document requires acknowledgement. Please acknowledge receipt:
{{acknowledgeLink}}
{{/if}}

Best regards,
The medi8 Team`,
    },
  ];

  for (const template of templates) {
    await prisma.notificationTemplate.upsert({
      where: { key: template.key },
      update: {},
      create: template,
    });
  }
  console.log('✅ Created notification templates');

  // Document templates
  const documentTemplates = [
    {
      name: 'Pre-Mediation Pack',
      description: 'Information pack sent to clients before mediation begins',
      category: 'AGREEMENT',
      requiresAcknowledgement: false,
      content: `# Pre-Mediation Information Pack

## Welcome to Mediation

Thank you for choosing mediation to resolve your dispute. This pack contains important information about the mediation process.

## Case Details
- Case Number: {{caseNumber}}
- Mediator: {{mediatorDisplayName}}
- Contact: {{mediatorPhone}}

## What to Expect

1. **Preparation**: Review all relevant documents and consider your goals
2. **The Session**: We'll meet to discuss the issues and explore solutions
3. **Agreement**: If we reach agreement, we'll document it in writing

## Mediation Principles

- **Confidential**: What's discussed stays in the room
- **Voluntary**: You're free to leave at any time
- **Impartial**: The mediator doesn't take sides
- **Self-Determination**: You control the outcome

## Next Steps

1. Review this pack carefully
2. Prepare any questions you have
3. Gather relevant documents
4. Attend the mediation session on time

If you have questions, please contact me at {{mediatorPhone}}.

---
*Generated on {{currentDate}}*`,
      createdBy: '', // Will be set by admin user ID in seed
      isActive: true,
    },
    {
      name: 'Fee Disclosure',
      description: 'Disclosure of mediator fees and costs',
      category: 'AGREEMENT',
      requiresAcknowledgement: true,
      content: `# Fee Disclosure Statement

**Case:** {{caseNumber}}  
**Mediator:** {{mediatorDisplayName}}  
**Date:** {{currentDate}}

## Mediation Fees

### Hourly Rate
- **Rate:** ${{hourlyRate}} per hour
- **Basis:** {{hourlyRateBasis}}
{{#if preMediationFee}}
- **Pre-mediation consultation:** ${{preMediationFee}}
{{/if}}

### How Fees Work

{{#if hourlyRateBasis equals "PER_PARTY"}}
The hourly rate of ${{hourlyRate}} applies to each party. For example, if there are 2 parties and the mediation takes 3 hours, the total fee would be ${{hourlyRate}} × 2 parties × 3 hours.
{{else}}
The hourly rate of ${{hourlyRate}} is a total rate for all parties combined, regardless of the number of parties involved.
{{/if}}

### Payment Terms

- Deposit: $250 (paid via platform)
- Balance: Invoiced after mediation
- Payment due: Within 7 days of invoice

### Additional Costs

- Travel may apply for locations beyond included distance
- Accommodation may apply for overnight stays
- All costs will be disclosed in advance

## Cancellation Policy

- More than 48 hours notice: Full refund of deposit
- Less than 48 hours notice: Deposit forfeited
- Same day cancellation: Full session fee may apply

## Questions?

If you have any questions about fees, please contact me before proceeding.

---

**By acknowledging this document, you confirm that you have read and understood the fee structure.**`,
      createdBy: '',
      isActive: true,
    },
    {
      name: 'Mediation Agreement',
      description: 'Agreement to mediate template',
      category: 'AGREEMENT',
      requiresAcknowledgement: true,
      content: `# Agreement to Mediate

**Case Number:** {{caseNumber}}  
**Date:** {{currentDate}}  
**Mediator:** {{mediatorDisplayName}}  
**Parties:** {{partyNames}}

## 1. Agreement to Mediate

The parties agree to participate in mediation to attempt to resolve their dispute.

## 2. Role of the Mediator

The mediator:
- Will facilitate discussions between the parties
- Will not act as a judge or make decisions for the parties
- Will remain neutral and impartial
- Will not provide legal advice

## 3. Confidentiality

All communications during mediation are confidential and cannot be used in subsequent legal proceedings, except as required by law.

## 4. Voluntary Participation

Participation is voluntary. Any party may withdraw at any time.

## 5. Good Faith

The parties agree to participate in good faith and make reasonable efforts to resolve the dispute.

## 6. Fees and Costs

- Fees are as disclosed in the Fee Disclosure Statement
- Each party is responsible for their own legal costs (if applicable)

## 7. Authority to Settle

Each party confirms they have authority to settle the dispute or will seek such authority if needed.

## 8. Without Prejudice

All discussions are on a without prejudice basis.

## 9. Agreement

If an agreement is reached, it will be documented in writing and signed by all parties.

---

**By acknowledging this document, you agree to participate in mediation under these terms.**

Mediator: {{mediatorDisplayName}}  
Contact: {{mediatorPhone}}  
Date: {{currentDate}}`,
      createdBy: '',
      isActive: true,
    },
  ];

  // Get admin user ID for createdBy
  const adminUser = await prisma.user.findFirst({
    where: { role: 'ADMIN' },
  });

  if (adminUser) {
    for (const template of documentTemplates) {
      await prisma.documentTemplate.upsert({
        where: { 
          // Use composite unique constraint if you have one, or check by name
          name: template.name 
        },
        update: {
          content: template.content,
          description: template.description,
          isActive: template.isActive,
          requiresAcknowledgement: template.requiresAcknowledgement,
        },
        create: {
          ...template,
          createdBy: adminUser.id,
        },
      });
    }
    console.log('✅ Created document templates');
  }

  console.log('🎉 Seed completed successfully!');
  console.log('\n📝 Default admin credentials:');
  console.log('   Email: admin@medi8.com.au');
  console.log('   Password: admin123');
  console.log('\n⚠️  Please change the admin password after first login!\n');
}

main()
  .catch((e) => {
    console.error('❌ Seed failed:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
