import { getCurrentUser } from '@/modules/auth/actions';
import { redirect } from 'next/navigation';
import { PaymentProcessingService } from '@/modules/payment/processing/service';

const paymentService = new PaymentProcessingService();

export default async function AdminPaymentsPage() {
  const user = await getCurrentUser();
  
  if (!user || user.role !== 'ADMIN') {
    redirect('/login');
  }

  const stats = await paymentService.getPlatformStats();
  const recentPayments = await paymentService.getAllPayments();
  const last10 = recentPayments.slice(0, 10);

  return (
    <div>
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">Platform Payments</h1>
        <p className="text-gray-600 mt-2">
          Overview of all payment activity
        </p>
      </div>

      {/* Platform Revenue Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <div className="card">
          <div className="text-sm font-medium text-gray-600 mb-1">Total Processed</div>
          <div className="text-3xl font-bold text-gray-900">
            ${stats.totalProcessed.toFixed(2)}
          </div>
          <div className="text-xs text-gray-500 mt-1">
            {stats.totalPayments} transactions
          </div>
        </div>

        <div className="card">
          <div className="text-sm font-medium text-gray-600 mb-1">Platform Revenue</div>
          <div className="text-3xl font-bold text-green-600">
            ${stats.totalRevenue.toFixed(2)}
          </div>
          <div className="text-xs text-gray-500 mt-1">
            Deposits: ${stats.depositRevenue.toFixed(2)} | Balance: ${stats.balanceRevenue.toFixed(2)}
          </div>
        </div>

        <div className="card">
          <div className="text-sm font-medium text-gray-600 mb-1">To Mediators</div>
          <div className="text-3xl font-bold text-gray-900">
            ${stats.totalToMediators.toFixed(2)}
          </div>
          <div className="text-xs text-gray-500 mt-1">
            Via Stripe Connect
          </div>
        </div>
      </div>

      {/* Fee Breakdown */}
      <div className="card mb-8">
        <h2 className="text-xl font-semibold mb-4">Revenue Breakdown</h2>
        <div className="space-y-3">
          <div className="flex items-center justify-between pb-3 border-b">
            <div>
              <div className="font-medium">Deposit Fees ($250 each)</div>
              <div className="text-sm text-gray-600">Platform keeps 100% of deposits</div>
            </div>
            <div className="text-xl font-semibold text-green-600">
              ${stats.depositRevenue.toFixed(2)}
            </div>
          </div>
          <div className="flex items-center justify-between">
            <div>
              <div className="font-medium">Balance Payment Fees (10% + GST)</div>
              <div className="text-sm text-gray-600">Platform fee on hourly/balance payments</div>
            </div>
            <div className="text-xl font-semibold text-green-600">
              ${stats.balanceRevenue.toFixed(2)}
            </div>
          </div>
        </div>
      </div>

      {/* Recent Payments */}
      <div className="card">
        <h2 className="text-xl font-semibold mb-4">Recent Payments</h2>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50 border-b">
              <tr>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Case</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Mediator</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Client</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Gross</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Platform Fee</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">To Mediator</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {last10.map((payment) => (
                <tr key={payment.id} className="hover:bg-gray-50">
                  <td className="px-4 py-3 text-sm text-gray-900">
                    {new Date(payment.createdAt).toLocaleDateString()}
                  </td>
                  <td className="px-4 py-3 text-sm text-gray-900">
                    {payment.case.caseNumber}
                  </td>
                  <td className="px-4 py-3 text-sm text-gray-600">
                    {payment.mediator?.firstName} {payment.mediator?.lastName}
                  </td>
                  <td className="px-4 py-3 text-sm text-gray-600">
                    {payment.client.firstName} {payment.client.lastName}
                  </td>
                  <td className="px-4 py-3 text-sm">
                    <span className={`px-2 py-1 rounded text-xs font-medium ${
                      payment.type === 'DEPOSIT' ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800'
                    }`}>
                      {payment.type}
                    </span>
                  </td>
                  <td className="px-4 py-3 text-sm text-gray-900">
                    ${payment.stripeGrossAmount.toString()}
                  </td>
                  <td className="px-4 py-3 text-sm font-medium text-green-600">
                    ${(Number(payment.platformFeeAmount) + Number(payment.platformFeeGst)).toFixed(2)}
                  </td>
                  <td className="px-4 py-3 text-sm text-gray-900">
                    ${payment.mediatorAmount.toString()}
                  </td>
                  <td className="px-4 py-3 text-sm">
                    <span className={`px-2 py-1 rounded text-xs font-medium ${
                      payment.status === 'SUCCEEDED' ? 'bg-green-100 text-green-800' :
                      payment.status === 'PENDING' ? 'bg-yellow-100 text-yellow-800' :
                      'bg-red-100 text-red-800'
                    }`}>
                      {payment.status}
                    </span>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
}
