import { getCurrentUser } from '@/modules/auth/actions';
import { redirect } from 'next/navigation';
import { CaseService } from '@/modules/case/service';
import Link from 'next/link';

const caseService = new CaseService();

export default async function ClientCasesPage() {
  const user = await getCurrentUser();
  
  if (!user?.client) {
    redirect('/login');
  }

  const cases = await caseService.getClientCases(user.client.id);

  const statusColors = {
    INQUIRY: 'bg-yellow-100 text-yellow-800',
    QUOTE_SENT: 'bg-blue-100 text-blue-800',
    BOOKED: 'bg-purple-100 text-purple-800',
    IN_PROGRESS: 'bg-green-100 text-green-800',
    COMPLETED: 'bg-gray-100 text-gray-800',
    CLOSED: 'bg-gray-100 text-gray-600',
    CANCELLED: 'bg-red-100 text-red-800',
  };

  return (
    <div>
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">My Cases</h1>
        <p className="text-gray-600 mt-2">
          View and track your mediation cases
        </p>
      </div>

      {cases.length === 0 ? (
        <div className="card text-center py-12">
          <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
          <p className="text-gray-600 mb-4">
            You don't have any cases yet
          </p>
          <Link href="/practices" className="btn-primary inline-block">
            Find a Mediator
          </Link>
        </div>
      ) : (
        <div className="space-y-4">
          {cases.map(caseRecord => (
            <div key={caseRecord.id} className="card">
              <div className="flex items-start justify-between mb-4">
                <div className="flex-1">
                  <h3 className="text-lg font-semibold mb-1">{caseRecord.title}</h3>
                  <p className="text-sm text-gray-600">
                    {caseRecord.caseNumber} • With {caseRecord.mediator.firstName} {caseRecord.mediator.lastName}
                  </p>
                </div>
                <span className={`px-3 py-1 rounded-full text-sm font-medium ${statusColors[caseRecord.status]}`}>
                  {caseRecord.status.replace('_', ' ')}
                </span>
              </div>

              {caseRecord.description && (
                <p className="text-sm text-gray-700 mb-4 line-clamp-2">
                  {caseRecord.description}
                </p>
              )}

              <div className="flex items-center justify-between">
                <div className="text-sm text-gray-600">
                  {caseRecord.disputeType && <span>{caseRecord.disputeType}</span>}
                  {caseRecord.estimatedHours && (
                    <span> • {caseRecord.estimatedHours}hrs estimated</span>
                  )}
                </div>
                <div className="text-xs text-gray-500">
                  Created {new Date(caseRecord.createdAt).toLocaleDateString()}
                </div>
              </div>

              {caseRecord.status === 'INQUIRY' && (
                <div className="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded text-sm text-yellow-800">
                  Your inquiry has been submitted. The mediator will review and respond shortly.
                </div>
              )}

              {caseRecord.status === 'QUOTE_SENT' && (
                <div className="mt-4 p-3 bg-blue-50 border border-blue-200 rounded text-sm text-blue-800">
                  The mediator has sent you a quote. Payment functionality coming in Milestone 5.
                </div>
              )}
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
