import { getCurrentUser } from '@/modules/auth/actions';
import { redirect } from 'next/navigation';
import { CaseService } from '@/modules/case/service';
import { CaseCard } from '@/components/case/CaseCard';
import Link from 'next/link';

const caseService = new CaseService();

export default async function MediatorCasesPage({
  searchParams,
}: {
  searchParams: { status?: string };
}) {
  const user = await getCurrentUser();
  
  if (!user?.mediator) {
    redirect('/login');
  }

  const statusFilter = searchParams.status as any;
  const cases = await caseService.getMediatorCases(user.mediator.id, statusFilter);
  const stats = await caseService.getMediatorStats(user.mediator.id);

  return (
    <div>
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">My Cases</h1>
        <p className="text-gray-600 mt-2">
          Manage your mediation cases
        </p>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="card">
          <div className="text-sm text-gray-600">Total Cases</div>
          <div className="text-2xl font-bold">{stats.total}</div>
        </div>
        <div className="card">
          <div className="text-sm text-gray-600">New Inquiries</div>
          <div className="text-2xl font-bold text-orange-600">{stats.inquiries}</div>
        </div>
        <div className="card">
          <div className="text-sm text-gray-600">Active Cases</div>
          <div className="text-2xl font-bold text-blue-600">{stats.active}</div>
        </div>
        <div className="card">
          <div className="text-sm text-gray-600">Completed</div>
          <div className="text-2xl font-bold text-green-600">{stats.completed}</div>
        </div>
      </div>

      {/* Filters */}
      <div className="mb-6 flex items-center space-x-2">
        <Link
          href="/mediator/cases"
          className={`px-4 py-2 rounded-lg text-sm font-medium ${
            !statusFilter
              ? 'bg-primary-600 text-white'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          All
        </Link>
        <Link
          href="/mediator/cases?status=INQUIRY"
          className={`px-4 py-2 rounded-lg text-sm font-medium ${
            statusFilter === 'INQUIRY'
              ? 'bg-primary-600 text-white'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          Inquiries
        </Link>
        <Link
          href="/mediator/cases?status=BOOKED"
          className={`px-4 py-2 rounded-lg text-sm font-medium ${
            statusFilter === 'BOOKED'
              ? 'bg-primary-600 text-white'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          Booked
        </Link>
        <Link
          href="/mediator/cases?status=IN_PROGRESS"
          className={`px-4 py-2 rounded-lg text-sm font-medium ${
            statusFilter === 'IN_PROGRESS'
              ? 'bg-primary-600 text-white'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          In Progress
        </Link>
        <Link
          href="/mediator/cases?status=COMPLETED"
          className={`px-4 py-2 rounded-lg text-sm font-medium ${
            statusFilter === 'COMPLETED'
              ? 'bg-primary-600 text-white'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          Completed
        </Link>
      </div>

      {/* Cases List */}
      {cases.length === 0 ? (
        <div className="card text-center py-12">
          <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
          <p className="text-gray-600 mb-4">
            {statusFilter ? `No ${statusFilter.toLowerCase()} cases` : 'No cases yet'}
          </p>
          <p className="text-sm text-gray-500">
            Cases will appear here when clients submit inquiries
          </p>
        </div>
      ) : (
        <div className="space-y-4">
          {cases.map(caseRecord => (
            <CaseCard key={caseRecord.id} case={caseRecord} />
          ))}
        </div>
      )}
    </div>
  );
}
