import { getCurrentUser } from '@/modules/auth/actions';
import { redirect } from 'next/navigation';
import { MediatorProfileService } from '@/modules/mediator/profile/service';
import { CaseService } from '@/modules/case/service';
import Link from 'next/link';

const profileService = new MediatorProfileService();
const caseService = new CaseService();

export default async function MediatorDashboardPage() {
  const user = await getCurrentUser();
  
  if (!user?.mediator) {
    redirect('/login');
  }

  const mediator = user.mediator;
  const isComplete = await profileService.isProfileComplete(mediator.id);
  const isOnTrial = mediator.subscriptionStatus === 'TRIAL';
  const stats = await caseService.getMediatorStats(mediator.id);

  // Calculate days remaining in trial
  let trialDaysRemaining = 0;
  if (isOnTrial && mediator.subscriptionEndsAt) {
    const now = new Date();
    const trialEnd = new Date(mediator.subscriptionEndsAt);
    trialDaysRemaining = Math.ceil((trialEnd.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
  }

  return (
    <div>
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">
          Welcome, {mediator.firstName}!
        </h1>
        <p className="text-gray-600 mt-2">
          Manage your mediation practice
        </p>
      </div>

      {/* Trial Status Banner */}
      {isOnTrial && trialDaysRemaining > 0 && (
        <div className="mb-6 bg-blue-50 border border-blue-200 rounded-lg p-4">
          <div className="flex items-start">
            <svg className="w-5 h-5 text-blue-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
            </svg>
            <div className="flex-1">
              <div className="font-medium text-blue-900">
                Trial Active: {trialDaysRemaining} days remaining
              </div>
              <p className="text-sm text-blue-800 mt-1">
                You're currently on a free trial. After your trial ends, plans start at $55/month (inc GST).
              </p>
            </div>
          </div>
        </div>
      )}

      {/* Profile Status */}
      <div className="mb-6">
        {mediator.status === 'PENDING_APPROVAL' ? (
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex items-start">
              <svg className="w-5 h-5 text-yellow-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
              </svg>
              <div>
                <div className="font-medium text-yellow-900">Profile Under Review</div>
                <p className="text-sm text-yellow-800 mt-1">
                  Your profile has been submitted and is being reviewed by our team.
                </p>
              </div>
            </div>
          </div>
        ) : mediator.status === 'APPROVED' ? (
          <div className="bg-green-50 border border-green-200 rounded-lg p-4">
            <div className="flex items-start">
              <svg className="w-5 h-5 text-green-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
              </svg>
              <div>
                <div className="font-medium text-green-900">Profile Approved!</div>
                <p className="text-sm text-green-800 mt-1">
                  Your profile has been approved and will be published shortly.
                </p>
              </div>
            </div>
          </div>
        ) : mediator.status === 'PUBLISHED' ? (
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <div className="flex items-start">
              <svg className="w-5 h-5 text-blue-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
              </svg>
              <div className="flex-1 flex items-start justify-between">
                <div>
                  <div className="font-medium text-blue-900">Profile Published</div>
                  <p className="text-sm text-blue-800 mt-1">
                    Your profile is live and visible to potential clients.
                  </p>
                </div>
                <Link 
                  href={`/practices/${mediator.slug}`}
                  target="_blank"
                  className="text-sm text-blue-600 hover:text-blue-700 font-medium whitespace-nowrap ml-4"
                >
                  View public page →
                </Link>
              </div>
            </div>
          </div>
        ) : mediator.status === 'REJECTED' ? (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
            <div className="flex items-start">
              <svg className="w-5 h-5 text-red-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
              </svg>
              <div>
                <div className="font-medium text-red-900">Profile Not Approved</div>
                {mediator.approvalNotes && (
                  <p className="text-sm text-red-800 mt-1">
                    <strong>Reason:</strong> {mediator.approvalNotes}
                  </p>
                )}
                <p className="text-sm text-red-800 mt-2">
                  Please <Link href="/mediator/profile" className="underline">update your profile</Link> and resubmit.
                </p>
              </div>
            </div>
          </div>
        ) : (
          <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
            <div className="flex items-start justify-between">
              <div className="flex items-start">
                <svg className="w-5 h-5 text-gray-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                </svg>
                <div>
                  <div className="font-medium text-gray-900">Complete Your Profile</div>
                  <p className="text-sm text-gray-700 mt-1">
                    Fill in your profile information to get started
                  </p>
                </div>
              </div>
              <Link href="/mediator/profile" className="btn-primary whitespace-nowrap ml-4">
                Complete Profile
              </Link>
            </div>
          </div>
        )}
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <div className="card">
          <div className="text-sm font-medium text-gray-600 mb-1">
            Profile Status
          </div>
          <div className="text-2xl font-bold text-gray-900 capitalize">
            {mediator.status.toLowerCase().replace('_', ' ')}
          </div>
          {!isComplete && mediator.status === 'PENDING_APPROVAL' && (
            <div className="text-xs text-gray-500 mt-2">
              Complete all required fields
            </div>
          )}
        </div>

        <div className="card">
          <div className="text-sm font-medium text-gray-600 mb-1">
            Active Cases
          </div>
          <div className="text-2xl font-bold text-gray-900">{stats.active}</div>
          <div className="text-xs text-gray-500 mt-2">
            {stats.inquiries} new inquiries
          </div>
        </div>

        <div className="card">
          <div className="text-sm font-medium text-gray-600 mb-1">
            Subscription
          </div>
          <div className="text-2xl font-bold text-gray-900 capitalize">
            {mediator.subscriptionStatus.toLowerCase()}
          </div>
        </div>
      </div>

      {/* Quick Actions */}
      <div className="card">
        <h2 className="text-xl font-semibold mb-4">Quick Actions</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Link
            href="/mediator/profile"
            className="p-4 border border-gray-200 rounded-lg hover:border-primary-300 hover:bg-primary-50 transition-colors"
          >
            <div className="font-medium text-gray-900 mb-1">Edit Profile</div>
            <div className="text-sm text-gray-600">
              Update your bio, specializations, and experience
            </div>
          </Link>

          <Link
            href="/mediator/profile/pricing"
            className="p-4 border border-gray-200 rounded-lg hover:border-primary-300 hover:bg-primary-50 transition-colors"
          >
            <div className="font-medium text-gray-900 mb-1">Set Pricing</div>
            <div className="text-sm text-gray-600">
              Configure your rates and pricing policies
            </div>
          </Link>

          <Link
            href="/mediator/cases"
            className="p-4 border border-gray-200 rounded-lg hover:border-primary-300 hover:bg-primary-50 transition-colors"
          >
            <div className="font-medium text-gray-900 mb-1">Manage Cases</div>
            <div className="text-sm text-gray-600">
              View and manage your mediation cases
            </div>
            {stats.inquiries > 0 && (
              <div className="mt-2 text-xs font-medium text-orange-600">
                {stats.inquiries} new {stats.inquiries === 1 ? 'inquiry' : 'inquiries'}
              </div>
            )}
          </Link>
        </div>
      </div>
    </div>
  );
}
