import { redirect } from 'next/navigation';
import { getCurrentUser } from '@/modules/auth/actions';
import { MediatorProfileService } from '@/modules/mediator/profile/service';
import { MediatorPricingService } from '@/modules/mediator/pricing/service';
import { ProfileBasicInfo } from '@/components/mediator/ProfileBasicInfo';
import { ProfileSpecializations } from '@/components/mediator/ProfileSpecializations';
import { ProfilePricing } from '@/components/mediator/ProfilePricing';
import { ProfileCompletion } from '@/components/mediator/ProfileCompletion';

const profileService = new MediatorProfileService();
const pricingService = new MediatorPricingService();

export default async function MediatorProfileEditPage() {
  const user = await getCurrentUser();

  if (!user?.mediator) {
    redirect('/login');
  }

  const mediator = await profileService.getProfile(user.mediator.id);
  const pricing = await pricingService.getPricing(user.mediator.id);
  const completion = await profileService.getProfileCompletion(user.mediator.id);

  const canEdit = mediator.status !== 'SUSPENDED';

  return (
    <div>
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">Edit Profile</h1>
        <p className="text-gray-600 mt-2">
          Complete your profile to get approved and start receiving client inquiries
        </p>
      </div>

      {/* Profile Completion */}
      <ProfileCompletion
        percentage={completion.percentage}
        missingFields={completion.missingFields}
        status={mediator.status}
        mediatorId={mediator.id}
        userId={user.id}
      />

      {!canEdit && (
        <div className="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
          <div className="flex items-start">
            <svg className="w-5 h-5 text-red-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
            </svg>
            <div className="flex-1">
              <div className="font-medium text-red-900">
                Profile Suspended
              </div>
              <p className="text-sm text-red-800 mt-1">
                Your profile has been suspended. Please contact support for assistance.
              </p>
            </div>
          </div>
        </div>
      )}

      {/* Tabs */}
      <div className="border-b border-gray-200 mb-8">
        <nav className="-mb-px flex space-x-8">
          <a
            href="#basic"
            className="border-b-2 border-primary-500 py-4 px-1 text-sm font-medium text-primary-600"
          >
            Basic Info
          </a>
          <a
            href="#specializations"
            className="border-b-2 border-transparent py-4 px-1 text-sm font-medium text-gray-500 hover:text-gray-700 hover:border-gray-300"
          >
            Specializations
          </a>
          <a
            href="#pricing"
            className="border-b-2 border-transparent py-4 px-1 text-sm font-medium text-gray-500 hover:text-gray-700 hover:border-gray-300"
          >
            Pricing
          </a>
        </nav>
      </div>

      {/* Forms */}
      <div className="space-y-8">
        <ProfileBasicInfo
          mediator={mediator}
          userId={user.id}
          disabled={!canEdit}
        />

        <ProfileSpecializations
          mediator={mediator}
          userId={user.id}
          disabled={!canEdit}
        />

        <ProfilePricing
          mediator={mediator}
          pricing={pricing}
          userId={user.id}
          disabled={!canEdit}
        />
      </div>
    </div>
  );
}
