import { getCurrentUser } from '@/modules/auth/actions';
import { redirect } from 'next/navigation';
import { MediatorProfileService } from '@/modules/mediator/profile/service';
import { ProfileForm } from '@/components/mediator/ProfileForm';
import { SpecializationsForm } from '@/components/mediator/SpecializationsForm';
import { CertificationsForm } from '@/components/mediator/CertificationsForm';
import { LanguagesForm } from '@/components/mediator/LanguagesForm';
import { ProfileStatus } from '@/components/mediator/ProfileStatus';
import Link from 'next/link';

const profileService = new MediatorProfileService();

export default async function MediatorProfilePage() {
  const user = await getCurrentUser();
  
  if (!user?.mediator) {
    redirect('/login');
  }

  const mediator = await profileService.getProfile(user.mediator.id);
  const isComplete = await profileService.isProfileComplete(user.mediator.id);

  return (
    <div className="max-w-4xl mx-auto">
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">My Profile</h1>
        <p className="text-gray-600 mt-2">
          Manage your practice information and settings
        </p>
      </div>

      {/* Profile Status Banner */}
      <ProfileStatus mediator={mediator} isComplete={isComplete} />

      {/* Quick Actions */}
      <div className="mb-6 flex items-center space-x-4">
        <Link href="/mediator/profile/pricing" className="btn-secondary">
          Manage Pricing
        </Link>
        {mediator.status === 'PUBLISHED' && (
          <Link 
            href={`/practices/${mediator.slug}`}
            target="_blank"
            className="text-sm text-primary-600 hover:text-primary-700"
          >
            View public page →
          </Link>
        )}
      </div>

      {/* Profile Forms */}
      <div className="space-y-6">
        {/* Basic Information */}
        <ProfileForm mediator={mediator} />

        {/* Specializations */}
        <SpecializationsForm 
          specializations={mediator.specializations} 
        />

        {/* Certifications */}
        <CertificationsForm 
          certifications={mediator.certifications} 
        />

        {/* Languages */}
        <LanguagesForm 
          languages={mediator.languages} 
        />

        {/* GST Information */}
        <div className="card">
          <h2 className="text-xl font-semibold mb-4">GST Registration</h2>
          <div className="space-y-4">
            <div>
              <div className="label">Status</div>
              <div className="text-lg">
                {mediator.isGstRegistered ? (
                  <span className="text-green-600">✓ Registered for GST</span>
                ) : (
                  <span className="text-gray-600">Not registered</span>
                )}
              </div>
            </div>
            {mediator.isGstRegistered && mediator.gstNumber && (
              <div>
                <div className="label">ABN / GST Number</div>
                <div className="text-lg font-mono">{mediator.gstNumber}</div>
              </div>
            )}
            <p className="text-sm text-gray-600">
              GST registration affects how prices are displayed to clients.
              Contact support if you need to update your GST status.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
