import { headers } from 'next/headers';
import { NextResponse } from 'next/server';
import { stripe } from '@/lib/stripe';
import { prisma } from '@/lib/db';
import { PaymentProcessingService } from '@/modules/payment/processing/service';
import { sendEmail } from '@/lib/email';

const paymentService = new PaymentProcessingService();

export async function POST(req: Request) {
  const body = await req.text();
  const signature = headers().get('stripe-signature');

  if (!signature) {
    return NextResponse.json({ error: 'No signature' }, { status: 400 });
  }

  let event;

  try {
    event = stripe.webhooks.constructEvent(
      body,
      signature,
      process.env.STRIPE_WEBHOOK_SECRET!
    );
  } catch (err) {
    console.error('Webhook signature verification failed:', err);
    return NextResponse.json(
      { error: 'Webhook signature verification failed' },
      { status: 400 }
    );
  }

  // Idempotency check - don't process the same event twice
  const existingEvent = await prisma.webhookEvent.findUnique({
    where: { eventId: event.id },
  });

  if (existingEvent?.processed) {
    console.log('Event already processed:', event.id);
    return NextResponse.json({ received: true });
  }

  // Create webhook event record
  await prisma.webhookEvent.create({
    data: {
      provider: 'STRIPE',
      eventType: event.type,
      eventId: event.id,
      payload: event.data.object as any,
      processed: false,
    },
  });

  try {
    // Handle different event types
    switch (event.type) {
      case 'payment_intent.succeeded':
        await handlePaymentSuccess(event.data.object);
        break;

      case 'payment_intent.payment_failed':
        await handlePaymentFailed(event.data.object);
        break;

      case 'charge.refunded':
        await handleRefund(event.data.object);
        break;

      case 'account.updated':
        await handleAccountUpdated(event.data.object);
        break;

      default:
        console.log('Unhandled event type:', event.type);
    }

    // Mark event as processed
    await prisma.webhookEvent.update({
      where: { eventId: event.id },
      data: {
        processed: true,
        processedAt: new Date(),
      },
    });

    return NextResponse.json({ received: true });
  } catch (error) {
    console.error('Error processing webhook:', error);

    // Log error but don't fail the webhook
    await prisma.webhookEvent.update({
      where: { eventId: event.id },
      data: {
        errorMessage: error instanceof Error ? error.message : 'Unknown error',
        retryCount: { increment: 1 },
      },
    });

    // Return 200 so Stripe doesn't retry
    return NextResponse.json({ received: true, error: 'Processing failed' });
  }
}

/**
 * Handle successful payment
 */
async function handlePaymentSuccess(paymentIntent: any) {
  console.log('Payment succeeded:', paymentIntent.id);

  // Find payment record
  const payment = await paymentService.getPaymentByIntentId(paymentIntent.id);

  if (!payment) {
    console.error('Payment record not found for intent:', paymentIntent.id);
    return;
  }

  // Update payment status
  await paymentService.updatePaymentStatus(payment.id, 'SUCCEEDED', {
    chargeId: paymentIntent.charges?.data[0]?.id,
  });

  // If deposit, update case status
  if (payment.type === 'DEPOSIT') {
    await prisma.case.update({
      where: { id: payment.caseId },
      data: {
        status: 'BOOKED',
      },
    });

    // Add timeline entry
    await prisma.caseTimeline.create({
      data: {
        caseId: payment.caseId,
        event: 'Deposit Paid',
        description: `Deposit of $${payment.amount.toString()} received`,
        status: 'BOOKED',
      },
    });

    // Sync to HubSpot: Deposit paid
    try {
      const { HubSpotIntegrationService } = await import('@/modules/integration/hubspot/service');
      const hubspotService = new HubSpotIntegrationService();
      await hubspotService.syncDepositPaid(payment.caseId);
    } catch (error) {
      console.error('[HubSpot] Sync failed (non-blocking):', error);
    }
  }

  // If balance payment, sync to HubSpot
  if (payment.type === 'HOURLY' || payment.type === 'TRAVEL' || payment.type === 'ACCOMMODATION') {
    try {
      const { HubSpotIntegrationService } = await import('@/modules/integration/hubspot/service');
      const hubspotService = new HubSpotIntegrationService();
      
      // Get total paid for this case
      const allPayments = await prisma.payment.findMany({
        where: {
          caseId: payment.caseId,
          status: 'SUCCEEDED',
        },
      });
      
      const totalPaid = allPayments.reduce(
        (sum, p) => sum + Number(p.stripeGrossAmount),
        0
      );
      
      await hubspotService.syncBalancePaid(payment.caseId, totalPaid);
    } catch (error) {
      console.error('[HubSpot] Sync failed (non-blocking):', error);
    }
  }

  // Send email notification to mediator
  const mediator = await prisma.mediator.findUnique({
    where: { id: payment.mediatorId },
    include: { user: true },
  });

  if (mediator) {
    await sendEmail({
      to: mediator.user.email,
      template: 'PAYMENT_RECEIVED',
      data: {
        firstName: mediator.firstName,
        amount: payment.amount.toString(),
        caseNumber: payment.case.caseNumber,
        caseId: payment.caseId,
      },
    });
  }

  console.log('Payment processed successfully:', payment.id);
}

/**
 * Handle failed payment
 */
async function handlePaymentFailed(paymentIntent: any) {
  console.log('Payment failed:', paymentIntent.id);

  const payment = await paymentService.getPaymentByIntentId(paymentIntent.id);

  if (!payment) {
    console.error('Payment record not found for intent:', paymentIntent.id);
    return;
  }

  await paymentService.updatePaymentStatus(payment.id, 'FAILED', {
    failureReason: paymentIntent.last_payment_error?.message,
  });

  // Notify client about failed payment
  const client = await prisma.client.findUnique({
    where: { id: payment.clientId },
    include: { user: true },
  });

  if (client) {
    // Note: Add PAYMENT_FAILED template to seed data
    console.log('Payment failed for client:', client.user.email);
  }
}

/**
 * Handle refund
 */
async function handleRefund(charge: any) {
  console.log('Refund processed:', charge.id);

  // Find payment by charge ID
  const payment = await prisma.payment.findUnique({
    where: { stripeChargeId: charge.id },
  });

  if (!payment) {
    console.error('Payment record not found for charge:', charge.id);
    return;
  }

  await paymentService.updatePaymentStatus(payment.id, 'REFUNDED', {
    refundAmount: charge.amount_refunded,
  });
}

/**
 * Handle Connect account updates
 */
async function handleAccountUpdated(account: any) {
  console.log('Account updated:', account.id);

  const mediator = await prisma.mediator.findUnique({
    where: { stripeAccountId: account.id },
  });

  if (!mediator) {
    console.error('Mediator not found for account:', account.id);
    return;
  }

  // Update onboarding status
  const isOnboarded = account.charges_enabled && account.details_submitted;

  if (isOnboarded && !mediator.stripeOnboarded) {
    await prisma.mediator.update({
      where: { id: mediator.id },
      data: {
        stripeOnboarded: true,
        stripeOnboardedAt: new Date(),
      },
    });

    console.log('Mediator onboarding completed:', mediator.id);
  }
}
