import { MediatorSearchService } from '@/modules/mediator/search/service';
import { SearchFilters } from '@/components/search/SearchFilters';
import { PriceEstimator } from '@/components/search/PriceEstimator';
import { SearchResults } from '@/components/search/SearchResults';
import Link from 'next/link';

const searchService = new MediatorSearchService();

interface SearchParams {
  specialization?: string;
  minPrice?: string;
  maxPrice?: string;
  maxBudget?: string;
  minExperience?: string;
  search?: string;
}

export default async function FindMediatorPage({
  searchParams,
}: {
  searchParams: SearchParams;
}) {
  // Parse search params
  const filters = {
    specialization: searchParams.specialization,
    minPrice: searchParams.minPrice ? parseFloat(searchParams.minPrice) : undefined,
    maxPrice: searchParams.maxPrice ? parseFloat(searchParams.maxPrice) : undefined,
    maxBudget: searchParams.maxBudget ? parseFloat(searchParams.maxBudget) : undefined,
    minExperience: searchParams.minExperience ? parseInt(searchParams.minExperience) : undefined,
    search: searchParams.search,
  };

  // Get initial data
  const mediators = await searchService.searchMediators(filters);
  const priceRanges = await searchService.getPriceRanges();
  const specializations = await searchService.getAvailableSpecializations();
  const languages = await searchService.getAvailableLanguages();

  const hasFilters = Object.values(filters).some(v => v !== undefined);

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 flex items-center justify-between">
          <Link href="/" className="flex items-center space-x-2">
            <div className="w-10 h-10 bg-primary-600 rounded-lg flex items-center justify-center">
              <span className="text-white font-bold text-xl">M8</span>
            </div>
            <span className="text-2xl font-bold text-gray-900">medi8</span>
          </Link>
          <div className="flex items-center space-x-4">
            <Link href="/login" className="text-gray-600 hover:text-gray-900">
              Login
            </Link>
            <Link href="/register/client" className="btn-primary">
              Get Started
            </Link>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="mb-8">
          <h1 className="text-4xl font-bold text-gray-900 mb-2">
            Find a Mediator
          </h1>
          <p className="text-xl text-gray-600">
            Search our directory of qualified mediators
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
          {/* Filters Sidebar */}
          <div className="lg:col-span-1">
            <SearchFilters
              filters={filters}
              priceRanges={priceRanges}
              specializations={specializations}
              languages={languages}
            />

            {/* Price Estimator */}
            <div className="mt-6">
              <PriceEstimator />
            </div>
          </div>

          {/* Results */}
          <div className="lg:col-span-3">
            <div className="mb-6 flex items-center justify-between">
              <div className="text-sm text-gray-600">
                {mediators.length} mediator{mediators.length !== 1 ? 's' : ''} found
                {hasFilters && (
                  <Link
                    href="/find-mediator"
                    className="ml-4 text-primary-600 hover:text-primary-700"
                  >
                    Clear filters
                  </Link>
                )}
              </div>
            </div>

            <SearchResults mediators={mediators} />
          </div>
        </div>
      </main>
    </div>
  );
}
