'use client';

import { Mediator, User, MediatorPricing } from '@prisma/client';
import { approveMediatorAction, rejectMediatorAction, publishMediatorAction } from '@/modules/mediator/profile/actions';
import { useState } from 'react';

type MediatorWithRelations = Mediator & {
  user: Pick<User, 'email' | 'createdAt'>;
  pricing: MediatorPricing | null;
  _count: { cases: number };
};

export function ApprovalCard({ mediator }: { mediator: MediatorWithRelations }) {
  const [isProcessing, setIsProcessing] = useState(false);
  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [rejectReason, setRejectReason] = useState('');
  const [approvalNotes, setApprovalNotes] = useState('');
  const [error, setError] = useState<string | null>(null);

  const handleApprove = async () => {
    setIsProcessing(true);
    setError(null);
    const result = await approveMediatorAction(mediator.id, approvalNotes);
    if (result.error) setError(result.error);
    setIsProcessing(false);
  };

  const handleReject = async () => {
    if (!rejectReason.trim()) {
      setError('Rejection reason is required');
      return;
    }
    setIsProcessing(true);
    setError(null);
    const result = await rejectMediatorAction(mediator.id, rejectReason);
    if (result.error) setError(result.error);
    else setShowRejectDialog(false);
    setIsProcessing(false);
  };

  const handlePublish = async () => {
    setIsProcessing(true);
    setError(null);
    const result = await publishMediatorAction(mediator.id);
    if (result.error) setError(result.error);
    setIsProcessing(false);
  };

  return (
    <div className="card">
      <div className="flex items-start justify-between mb-4">
        <div>
          <h3 className="text-xl font-semibold">
            {mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
          </h3>
          <p className="text-sm text-gray-600">{mediator.user.email}</p>
          <p className="text-xs text-gray-500 mt-1">
            Applied {new Date(mediator.user.createdAt).toLocaleDateString()}
          </p>
        </div>
        <span className={`px-3 py-1 rounded-full text-sm font-medium ${
          mediator.status === 'PENDING_APPROVAL' ? 'bg-yellow-100 text-yellow-800' :
          mediator.status === 'APPROVED' ? 'bg-green-100 text-green-800' :
          'bg-gray-100 text-gray-800'
        }`}>
          {mediator.status.replace('_', ' ')}
        </span>
      </div>

      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
          {error}
        </div>
      )}

      <div className="grid grid-cols-2 gap-6 mb-6">
        <div>
          <div className="text-sm font-medium text-gray-600 mb-2">Contact</div>
          <div className="space-y-1">
            <p className="text-sm">{mediator.phone || 'No phone'}</p>
            {mediator.officeAddress && (
              <p className="text-sm text-gray-600">{mediator.officeAddress}</p>
            )}
          </div>
        </div>

        <div>
          <div className="text-sm font-medium text-gray-600 mb-2">Experience</div>
          <div className="space-y-1">
            <p className="text-sm">{mediator.yearsExperience || 'Not specified'} years</p>
            <p className="text-sm text-gray-600">
              {mediator.languages.length} language{mediator.languages.length !== 1 ? 's' : ''}
            </p>
          </div>
        </div>
      </div>

      {mediator.specializations.length > 0 && (
        <div className="mb-4">
          <div className="text-sm font-medium text-gray-600 mb-2">Specializations</div>
          <div className="flex flex-wrap gap-2">
            {mediator.specializations.map(spec => (
              <span key={spec} className="px-2 py-1 bg-blue-100 text-blue-800 rounded text-sm">
                {spec}
              </span>
            ))}
          </div>
        </div>
      )}

      {mediator.certifications.length > 0 && (
        <div className="mb-4">
          <div className="text-sm font-medium text-gray-600 mb-2">Certifications</div>
          <ul className="list-disc list-inside text-sm space-y-1">
            {mediator.certifications.map((cert, i) => (
              <li key={i}>{cert}</li>
            ))}
          </ul>
        </div>
      )}

      {mediator.bio && (
        <div className="mb-4">
          <div className="text-sm font-medium text-gray-600 mb-2">Bio</div>
          <p className="text-sm text-gray-700 whitespace-pre-line line-clamp-3">
            {mediator.bio}
          </p>
        </div>
      )}

      {mediator.pricing && (
        <div className="mb-4 p-3 bg-gray-50 rounded">
          <div className="text-sm font-medium text-gray-600 mb-2">Pricing</div>
          <div className="space-y-1 text-sm">
            <p>
              <strong>Hourly:</strong> ${mediator.pricing.hourlyRate.toString()}/hr
              {mediator.pricing.hourlyRateBasis === 'PER_PARTY' ? ' per party' : ' total'}
              {mediator.isGstRegistered && ' (inc GST)'}
            </p>
            {mediator.pricing.preMediationFee && (
              <p>
                <strong>Pre-mediation fee:</strong> ${mediator.pricing.preMediationFee.toString()}
              </p>
            )}
          </div>
        </div>
      )}

      <div className="mb-4 border-t pt-4">
        <label className="label">Admin Notes (optional)</label>
        <textarea
          value={approvalNotes}
          onChange={e => setApprovalNotes(e.target.value)}
          className="input"
          rows={2}
          placeholder="Internal notes about this approval..."
        />
      </div>

      <div className="flex items-center space-x-3">
        {mediator.status === 'PENDING_APPROVAL' && (
          <>
            <button
              onClick={handleApprove}
              disabled={isProcessing}
              className="btn-primary"
            >
              {isProcessing ? 'Processing...' : 'Approve'}
            </button>
            <button
              onClick={() => setShowRejectDialog(true)}
              disabled={isProcessing}
              className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50"
            >
              Reject
            </button>
          </>
        )}

        {mediator.status === 'APPROVED' && (
          <button
            onClick={handlePublish}
            disabled={isProcessing}
            className="btn-primary"
          >
            {isProcessing ? 'Publishing...' : 'Publish Profile'}
          </button>
        )}
      </div>

      {showRejectDialog && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full">
            <h3 className="text-lg font-semibold mb-4">Reject Application</h3>
            <label className="label">Reason for rejection *</label>
            <textarea
              value={rejectReason}
              onChange={e => setRejectReason(e.target.value)}
              className="input mb-4"
              rows={4}
              placeholder="Explain why this application is being rejected..."
              required
            />
            <div className="flex items-center space-x-3">
              <button onClick={handleReject} disabled={isProcessing} className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                {isProcessing ? 'Rejecting...' : 'Confirm Rejection'}
              </button>
              <button onClick={() => setShowRejectDialog(false)} className="btn-secondary">
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
