'use client';

import { useState } from 'react';
import {
  approveMediatorAction,
  rejectMediatorAction,
  publishMediatorAction,
  suspendMediatorAction,
  unsuspendMediatorAction,
} from '@/modules/admin/approvals/actions';

interface MediatorReviewActionsProps {
  mediatorId: string;
  adminId: string;
  status: string;
}

export function MediatorReviewActions({
  mediatorId,
  adminId,
  status,
}: MediatorReviewActionsProps) {
  const [action, setAction] = useState<'approve' | 'reject' | 'suspend' | null>(null);
  const [reason, setReason] = useState('');
  const [notes, setNotes] = useState('');
  const [submitting, setSubmitting] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleApprove = async () => {
    setSubmitting(true);
    setError(null);

    const result = await approveMediatorAction(mediatorId, adminId, notes || undefined);

    if (result.error) {
      setError(result.error);
      setSubmitting(false);
    } else {
      window.location.reload();
    }
  };

  const handleReject = async () => {
    if (!reason.trim()) {
      setError('Rejection reason is required');
      return;
    }

    setSubmitting(true);
    setError(null);

    const formData = new FormData();
    formData.set('reason', reason);

    const result = await rejectMediatorAction(mediatorId, adminId, formData);

    if (result.error) {
      setError(result.error);
      setSubmitting(false);
    } else {
      window.location.reload();
    }
  };

  const handlePublish = async () => {
    setSubmitting(true);
    setError(null);

    const result = await publishMediatorAction(mediatorId, adminId);

    if (result.error) {
      setError(result.error);
      setSubmitting(false);
    } else {
      window.location.reload();
    }
  };

  const handleSuspend = async () => {
    if (!reason.trim()) {
      setError('Suspension reason is required');
      return;
    }

    setSubmitting(true);
    setError(null);

    const formData = new FormData();
    formData.set('reason', reason);

    const result = await suspendMediatorAction(mediatorId, adminId, formData);

    if (result.error) {
      setError(result.error);
      setSubmitting(false);
    } else {
      window.location.reload();
    }
  };

  const handleUnsuspend = async () => {
    setSubmitting(true);
    setError(null);

    const result = await unsuspendMediatorAction(mediatorId, adminId);

    if (result.error) {
      setError(result.error);
      setSubmitting(false);
    } else {
      window.location.reload();
    }
  };

  return (
    <div className="card">
      <h3 className="font-semibold text-gray-900 mb-4">Actions</h3>

      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
          {error}
        </div>
      )}

      {status === 'PENDING_APPROVAL' && !action && (
        <div className="space-y-3">
          <button
            onClick={() => setAction('approve')}
            className="w-full btn-primary"
          >
            Approve Profile
          </button>
          <button
            onClick={() => setAction('reject')}
            className="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700"
          >
            Reject Profile
          </button>
        </div>
      )}

      {status === 'PENDING_APPROVAL' && action === 'approve' && (
        <div className="space-y-4">
          <div>
            <label className="label">Notes (optional)</label>
            <textarea
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              rows={3}
              className="input"
              placeholder="Any notes for the approval..."
            />
          </div>

          <div className="flex space-x-2">
            <button
              onClick={handleApprove}
              disabled={submitting}
              className="flex-1 btn-primary"
            >
              {submitting ? 'Approving...' : 'Confirm Approval'}
            </button>
            <button
              onClick={() => setAction(null)}
              className="flex-1 btn-secondary"
              disabled={submitting}
            >
              Cancel
            </button>
          </div>
        </div>
      )}

      {status === 'PENDING_APPROVAL' && action === 'reject' && (
        <div className="space-y-4">
          <div>
            <label className="label">Rejection Reason *</label>
            <textarea
              value={reason}
              onChange={(e) => setReason(e.target.value)}
              rows={4}
              className="input"
              placeholder="Explain why the profile is being rejected..."
              required
            />
          </div>

          <div className="flex space-x-2">
            <button
              onClick={handleReject}
              disabled={submitting || !reason.trim()}
              className="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50"
            >
              {submitting ? 'Rejecting...' : 'Confirm Rejection'}
            </button>
            <button
              onClick={() => setAction(null)}
              className="flex-1 btn-secondary"
              disabled={submitting}
            >
              Cancel
            </button>
          </div>
        </div>
      )}

      {status === 'APPROVED' && (
        <div className="space-y-3">
          <button
            onClick={handlePublish}
            disabled={submitting}
            className="w-full btn-primary"
          >
            {submitting ? 'Publishing...' : 'Publish Profile'}
          </button>
          <p className="text-xs text-gray-500">
            Publishing will make the profile visible on /practices/{'{'}slug{'}'}
          </p>
        </div>
      )}

      {status === 'PUBLISHED' && !action && (
        <div className="space-y-3">
          <div className="p-3 bg-green-50 border border-green-200 rounded-lg text-sm text-green-800">
            ✓ Profile is published and live
          </div>
          <button
            onClick={() => setAction('suspend')}
            className="w-full px-4 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700"
          >
            Suspend Profile
          </button>
        </div>
      )}

      {status === 'PUBLISHED' && action === 'suspend' && (
        <div className="space-y-4">
          <div>
            <label className="label">Suspension Reason *</label>
            <textarea
              value={reason}
              onChange={(e) => setReason(e.target.value)}
              rows={4}
              className="input"
              placeholder="Explain why the profile is being suspended..."
              required
            />
          </div>

          <div className="flex space-x-2">
            <button
              onClick={handleSuspend}
              disabled={submitting || !reason.trim()}
              className="flex-1 px-4 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700 disabled:opacity-50"
            >
              {submitting ? 'Suspending...' : 'Confirm Suspension'}
            </button>
            <button
              onClick={() => setAction(null)}
              className="flex-1 btn-secondary"
              disabled={submitting}
            >
              Cancel
            </button>
          </div>
        </div>
      )}

      {status === 'SUSPENDED' && (
        <div className="space-y-3">
          <div className="p-3 bg-orange-50 border border-orange-200 rounded-lg text-sm text-orange-800">
            ⚠ Profile is suspended
          </div>
          <button
            onClick={handleUnsuspend}
            disabled={submitting}
            className="w-full btn-primary"
          >
            {submitting ? 'Unsuspending...' : 'Unsuspend Profile'}
          </button>
        </div>
      )}

      {status === 'REJECTED' && (
        <div className="p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
          Profile has been rejected
        </div>
      )}
    </div>
  );
}
