'use client';

import { Case, CaseParty, Client, User } from '@prisma/client';
import Link from 'next/link';

type CaseWithRelations = Case & {
  parties: (CaseParty & {
    client: Client & {
      user: { email: string };
    };
  })[];
  _count: {
    documents: number;
    payments: number;
    timeline: number;
  };
};

export function CaseCard({ case: caseRecord }: { case: CaseWithRelations }) {
  const primaryParty = caseRecord.parties.find(p => p.isPrimary);
  const statusColors = {
    INQUIRY: 'bg-yellow-100 text-yellow-800',
    QUOTE_SENT: 'bg-blue-100 text-blue-800',
    BOOKED: 'bg-purple-100 text-purple-800',
    IN_PROGRESS: 'bg-green-100 text-green-800',
    COMPLETED: 'bg-gray-100 text-gray-800',
    CLOSED: 'bg-gray-100 text-gray-600',
    CANCELLED: 'bg-red-100 text-red-800',
  };

  return (
    <Link href={`/mediator/cases/${caseRecord.id}`} className="card hover:shadow-lg transition-shadow">
      <div className="flex items-start justify-between mb-4">
        <div className="flex-1">
          <h3 className="text-lg font-semibold mb-1">{caseRecord.title}</h3>
          <p className="text-sm text-gray-600">
            {caseRecord.caseNumber} • {primaryParty ? `${primaryParty.client.firstName} ${primaryParty.client.lastName}` : 'No client'}
          </p>
        </div>
        <span className={`px-3 py-1 rounded-full text-sm font-medium ${statusColors[caseRecord.status]}`}>
          {caseRecord.status.replace('_', ' ')}
        </span>
      </div>

      {caseRecord.description && (
        <p className="text-sm text-gray-700 mb-4 line-clamp-2">
          {caseRecord.description}
        </p>
      )}

      <div className="flex items-center justify-between text-sm text-gray-600">
        <div className="flex items-center space-x-4">
          {caseRecord.disputeType && (
            <span>{caseRecord.disputeType}</span>
          )}
          {caseRecord.estimatedHours && (
            <span>• {caseRecord.estimatedHours}hrs estimated</span>
          )}
        </div>
        <div className="flex items-center space-x-3">
          {caseRecord._count.documents > 0 && (
            <span className="flex items-center">
              <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
              </svg>
              {caseRecord._count.documents}
            </span>
          )}
          {caseRecord._count.timeline > 0 && (
            <span className="flex items-center">
              <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              {caseRecord._count.timeline}
            </span>
          )}
        </div>
      </div>

      <div className="mt-4 text-xs text-gray-500">
        Created {new Date(caseRecord.createdAt).toLocaleDateString()}
      </div>
    </Link>
  );
}
