'use client';

import { Case, CaseTimeline as PrismaCaseTimeline, CaseStatus } from '@prisma/client';
import { useState } from 'react';
import { updateCaseStatusAction, addCaseNoteAction } from '@/modules/case/actions';
import Link from 'next/link';

type CaseWithRelations = Case & {
  parties: any[];
  mediator: any;
  timeline: PrismaCaseTimeline[];
};

export function CaseHeader({ case: caseRecord }: { case: CaseWithRelations }) {
  const statusColors = {
    INQUIRY: 'bg-yellow-100 text-yellow-800',
    QUOTE_SENT: 'bg-blue-100 text-blue-800',
    BOOKED: 'bg-purple-100 text-purple-800',
    IN_PROGRESS: 'bg-green-100 text-green-800',
    COMPLETED: 'bg-gray-100 text-gray-800',
    CLOSED: 'bg-gray-100 text-gray-600',
    CANCELLED: 'bg-red-100 text-red-800',
  };

  const primaryParty = caseRecord.parties.find(p => p.isPrimary);

  return (
    <div className="mb-6">
      <Link href="/mediator/cases" className="text-sm text-primary-600 hover:text-primary-700 mb-2 inline-block">
        ← Back to Cases
      </Link>
      <div className="flex items-start justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">{caseRecord.title}</h1>
          <div className="flex items-center space-x-3 mt-2 text-sm text-gray-600">
            <span className="font-mono">{caseRecord.caseNumber}</span>
            {primaryParty && (
              <>
                <span>•</span>
                <span>{primaryParty.client.firstName} {primaryParty.client.lastName}</span>
                <span>•</span>
                <span>{primaryParty.client.user.email}</span>
              </>
            )}
          </div>
        </div>
        <span className={`px-4 py-2 rounded-full text-sm font-medium ${statusColors[caseRecord.status]}`}>
          {caseRecord.status.replace('_', ' ')}
        </span>
      </div>
    </div>
  );
}

export function CaseDetails({ case: caseRecord }: { case: CaseWithRelations }) {
  return (
    <div className="card">
      <h2 className="text-xl font-semibold mb-4">Case Details</h2>
      <div className="space-y-4">
        {caseRecord.description && (
          <div>
            <div className="text-sm font-medium text-gray-600 mb-1">Description</div>
            <p className="text-gray-700 whitespace-pre-line">{caseRecord.description}</p>
          </div>
        )}
        
        <div className="grid grid-cols-2 gap-4">
          {caseRecord.disputeType && (
            <div>
              <div className="text-sm font-medium text-gray-600 mb-1">Dispute Type</div>
              <p className="text-gray-900">{caseRecord.disputeType}</p>
            </div>
          )}
          {caseRecord.estimatedHours && (
            <div>
              <div className="text-sm font-medium text-gray-600 mb-1">Estimated Hours</div>
              <p className="text-gray-900">{caseRecord.estimatedHours}</p>
            </div>
          )}
          {caseRecord.requiresTravel && (
            <div>
              <div className="text-sm font-medium text-gray-600 mb-1">Travel Required</div>
              <p className="text-gray-900">
                Yes {caseRecord.travelDistance && `(${caseRecord.travelDistance}km)`}
              </p>
            </div>
          )}
          {caseRecord.requiresAccommodation && (
            <div>
              <div className="text-sm font-medium text-gray-600 mb-1">Accommodation</div>
              <p className="text-gray-900">Required</p>
            </div>
          )}
        </div>

        {caseRecord.privateNotes && (
          <div className="border-t pt-4">
            <div className="text-sm font-medium text-gray-600 mb-1">Private Notes</div>
            <p className="text-gray-700 whitespace-pre-line">{caseRecord.privateNotes}</p>
          </div>
        )}
      </div>
    </div>
  );
}

export function CaseTimeline({ timeline }: { timeline: PrismaCaseTimeline[] }) {
  return (
    <div className="card">
      <h2 className="text-xl font-semibold mb-4">Timeline</h2>
      <div className="space-y-4">
        {timeline.map((entry, idx) => (
          <div key={entry.id} className="flex">
            <div className="flex flex-col items-center mr-4">
              <div className="w-3 h-3 bg-primary-600 rounded-full" />
              {idx < timeline.length - 1 && (
                <div className="w-0.5 h-full bg-gray-200 mt-2" />
              )}
            </div>
            <div className="flex-1 pb-4">
              <div className="font-medium text-gray-900">{entry.event}</div>
              {entry.description && (
                <p className="text-sm text-gray-600 mt-1">{entry.description}</p>
              )}
              <div className="text-xs text-gray-500 mt-2">
                {new Date(entry.createdAt).toLocaleString()}
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}

export function CaseNotes({ caseId }: { caseId: string }) {
  const [note, setNote] = useState('');
  const [isAdding, setIsAdding] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleAddNote = async () => {
    if (!note.trim()) return;
    setIsAdding(true);
    setError(null);

    const result = await addCaseNoteAction(caseId, note);
    
    if (result.error) {
      setError(result.error);
    } else {
      setNote('');
    }
    setIsAdding(false);
  };

  return (
    <div className="card">
      <h3 className="font-semibold mb-3">Add Note</h3>
      {error && (
        <div className="mb-3 p-2 bg-red-50 border border-red-200 rounded text-sm text-red-800">
          {error}
        </div>
      )}
      <textarea
        value={note}
        onChange={(e) => setNote(e.target.value)}
        className="input mb-3"
        rows={4}
        placeholder="Add a note to the timeline..."
        maxLength={2000}
      />
      <button
        onClick={handleAddNote}
        disabled={isAdding || !note.trim()}
        className="btn-primary w-full"
      >
        {isAdding ? 'Adding...' : 'Add Note'}
      </button>
    </div>
  );
}

export function CaseStatusManager({ case: caseRecord }: { case: CaseWithRelations }) {
  const [isUpdating, setIsUpdating] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const nextStatuses: Record<CaseStatus, CaseStatus[]> = {
    INQUIRY: ['QUOTE_SENT', 'CANCELLED'],
    QUOTE_SENT: ['BOOKED', 'CANCELLED'],
    BOOKED: ['IN_PROGRESS', 'CANCELLED'],
    IN_PROGRESS: ['COMPLETED', 'CANCELLED'],
    COMPLETED: ['CLOSED'],
    CLOSED: [],
    CANCELLED: [],
  };

  const handleStatusChange = async (newStatus: CaseStatus) => {
    setIsUpdating(true);
    setError(null);

    const result = await updateCaseStatusAction(caseRecord.id, newStatus);
    
    if (result.error) {
      setError(result.error);
    }
    setIsUpdating(false);
  };

  const possibleNextStatuses = nextStatuses[caseRecord.status];

  if (possibleNextStatuses.length === 0) {
    return (
      <div className="card">
        <h3 className="font-semibold mb-3">Status</h3>
        <p className="text-sm text-gray-600">
          This case is {caseRecord.status.toLowerCase()}. No further status changes available.
        </p>
      </div>
    );
  }

  return (
    <div className="card">
      <h3 className="font-semibold mb-3">Update Status</h3>
      {error && (
        <div className="mb-3 p-2 bg-red-50 border border-red-200 rounded text-sm text-red-800">
          {error}
        </div>
      )}
      <div className="space-y-2">
        {possibleNextStatuses.map(status => (
          <button
            key={status}
            onClick={() => handleStatusChange(status)}
            disabled={isUpdating}
            className="w-full px-4 py-2 text-left text-sm border border-gray-200 rounded-lg hover:bg-gray-50 disabled:opacity-50"
          >
            → {status.replace('_', ' ')}
          </button>
        ))}
      </div>
    </div>
  );
}
