'use client';

import { useState } from 'react';
import { createInquiryAction } from '@/modules/case/actions';
import { DISPUTE_TYPES } from '@/modules/case/validation';
import { useRouter } from 'next/navigation';

interface InquiryFormProps {
  mediatorId: string;
}

export function InquiryForm({ mediatorId }: InquiryFormProps) {
  const router = useRouter();
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [showTravelFields, setShowTravelFields] = useState(false);

  const handleSubmit = async (formData: FormData) => {
    setIsSubmitting(true);
    setError(null);

    const result = await createInquiryAction(mediatorId, formData);

    if (result.error) {
      setError(result.error);
      setIsSubmitting(false);
    } else if (result.caseId) {
      // Redirect to client cases page
      router.push('/client/cases');
    }
  };

  return (
    <form action={handleSubmit} className="space-y-4">
      {error && (
        <div className="p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
          {error}
        </div>
      )}

      <div>
        <label htmlFor="title" className="label">
          Case Title *
        </label>
        <input
          type="text"
          id="title"
          name="title"
          required
          className="input"
          placeholder="e.g., Family Property Dispute"
          minLength={5}
          maxLength={200}
        />
        <p className="text-xs text-gray-500 mt-1">
          Brief title for your case
        </p>
      </div>

      <div>
        <label htmlFor="disputeType" className="label">
          Dispute Type
        </label>
        <select
          id="disputeType"
          name="disputeType"
          className="input"
        >
          <option value="">Select type (optional)</option>
          {DISPUTE_TYPES.map(type => (
            <option key={type} value={type}>{type}</option>
          ))}
        </select>
      </div>

      <div>
        <label htmlFor="description" className="label">
          Description *
        </label>
        <textarea
          id="description"
          name="description"
          required
          className="input"
          rows={6}
          placeholder="Please describe the situation and what you hope to achieve through mediation..."
          minLength={20}
          maxLength={2000}
        />
        <p className="text-xs text-gray-500 mt-1">
          Minimum 20 characters. Be as detailed as possible.
        </p>
      </div>

      <div>
        <label htmlFor="estimatedHours" className="label">
          Estimated Hours (optional)
        </label>
        <input
          type="number"
          id="estimatedHours"
          name="estimatedHours"
          className="input w-32"
          min="0"
          max="1000"
          step="0.5"
          placeholder="2.0"
        />
        <p className="text-xs text-gray-500 mt-1">
          If you have an idea of session length
        </p>
      </div>

      <div className="flex items-center">
        <input
          type="checkbox"
          id="requiresTravel"
          name="requiresTravel"
          value="true"
          className="mr-2"
          onChange={(e) => setShowTravelFields(e.target.checked)}
        />
        <label htmlFor="requiresTravel" className="text-sm text-gray-700">
          This case requires the mediator to travel
        </label>
      </div>

      {showTravelFields && (
        <div className="ml-6 space-y-4">
          <div>
            <label htmlFor="travelDistance" className="label">
              Travel Distance (km)
            </label>
            <input
              type="number"
              id="travelDistance"
              name="travelDistance"
              className="input w-32"
              min="0"
              max="5000"
              placeholder="100"
            />
          </div>

          <div className="flex items-center">
            <input
              type="checkbox"
              id="requiresAccommodation"
              name="requiresAccommodation"
              value="true"
              className="mr-2"
            />
            <label htmlFor="requiresAccommodation" className="text-sm text-gray-700">
              Overnight accommodation required
            </label>
          </div>
        </div>
      )}

      <div className="pt-4">
        <button
          type="submit"
          disabled={isSubmitting}
          className="btn-primary w-full"
        >
          {isSubmitting ? 'Submitting...' : 'Submit Inquiry'}
        </button>
        <p className="text-xs text-gray-500 mt-2 text-center">
          The mediator will review your inquiry and respond shortly
        </p>
      </div>
    </form>
  );
}
