'use client';

import { useState } from 'react';
import { createDocumentFromTemplateAction, sendDocumentToClientAction } from '@/modules/document/actions';

interface DocumentInstance {
  id: string;
  name: string;
  description?: string;
  category: string;
  requiresAcknowledgement: boolean;
  createdAt: Date;
  sentRecords?: any[];
  acknowledgements?: any[];
}

interface CaseDocumentsProps {
  caseId: string;
  documents: DocumentInstance[];
  templates: any[];
  parties: any[];
}

export function CaseDocuments({ caseId, documents, templates, parties }: CaseDocumentsProps) {
  const [selectedTemplate, setSelectedTemplate] = useState('');
  const [isCreating, setIsCreating] = useState(false);
  const [isSending, setIsSending] = useState<string | null>(null);
  const [selectedDoc, setSelectedDoc] = useState<string | null>(null);
  const [message, setMessage] = useState('');
  const [error, setError] = useState<string | null>(null);

  const handleCreateDocument = async () => {
    if (!selectedTemplate) return;

    setIsCreating(true);
    setError(null);

    const result = await createDocumentFromTemplateAction(selectedTemplate, caseId);

    if (result.error) {
      setError(result.error);
    } else {
      setSelectedTemplate('');
      window.location.reload();
    }

    setIsCreating(false);
  };

  const handleSendDocument = async (documentId: string) => {
    const clientIds = parties.map(p => p.clientId);
    
    setIsSending(documentId);
    setError(null);

    const result = await sendDocumentToClientAction(documentId, clientIds, message);

    if (result.error) {
      setError(result.error);
    } else {
      setMessage('');
      setSelectedDoc(null);
      window.location.reload();
    }

    setIsSending(null);
  };

  return (
    <div className="space-y-6">
      {/* Create Document from Template */}
      <div className="card">
        <h3 className="text-lg font-semibold mb-4">Create Document</h3>
        
        {error && (
          <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
            {error}
          </div>
        )}

        <div className="flex items-center space-x-3">
          <select
            value={selectedTemplate}
            onChange={e => setSelectedTemplate(e.target.value)}
            className="input flex-1"
            disabled={isCreating}
          >
            <option value="">Select a template...</option>
            {templates.map(template => (
              <option key={template.id} value={template.id}>
                {template.name}
              </option>
            ))}
          </select>

          <button
            onClick={handleCreateDocument}
            disabled={!selectedTemplate || isCreating}
            className="btn-primary whitespace-nowrap"
          >
            {isCreating ? 'Creating...' : 'Create'}
          </button>
        </div>
      </div>

      {/* Documents List */}
      <div className="card">
        <h3 className="text-lg font-semibold mb-4">Case Documents</h3>

        {documents.length === 0 ? (
          <p className="text-gray-500 text-center py-8">
            No documents yet. Create one from a template above.
          </p>
        ) : (
          <div className="space-y-4">
            {documents.map(doc => {
              const sentCount = doc.sentRecords?.length || 0;
              const ackCount = doc.acknowledgements?.length || 0;
              const requiresAck = doc.requiresAcknowledgement;
              const fullyAcked = requiresAck ? ackCount >= parties.length : true;

              return (
                <div key={doc.id} className="border rounded-lg p-4">
                  <div className="flex items-start justify-between mb-3">
                    <div>
                      <h4 className="font-semibold">{doc.name}</h4>
                      {doc.description && (
                        <p className="text-sm text-gray-600 mt-1">{doc.description}</p>
                      )}
                      <p className="text-xs text-gray-500 mt-1">
                        Created {new Date(doc.createdAt).toLocaleDateString()}
                      </p>
                    </div>

                    <span className={`px-2 py-1 rounded text-xs font-medium ${
                      doc.category === 'AGREEMENT' ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-800'
                    }`}>
                      {doc.category}
                    </span>
                  </div>

                  <div className="flex items-center justify-between text-sm">
                    <div className="flex items-center space-x-4">
                      {sentCount > 0 ? (
                        <span className="text-green-600">
                          ✓ Sent to {sentCount} {sentCount === 1 ? 'party' : 'parties'}
                        </span>
                      ) : (
                        <span className="text-gray-500">Not sent</span>
                      )}

                      {requiresAck && sentCount > 0 && (
                        fullyAcked ? (
                          <span className="text-green-600">
                            ✓ Fully acknowledged
                          </span>
                        ) : (
                          <span className="text-yellow-600">
                            {ackCount}/{parties.length} acknowledged
                          </span>
                        )
                      )}
                    </div>

                    {sentCount === 0 && (
                      <button
                        onClick={() => setSelectedDoc(doc.id)}
                        className="btn-secondary text-sm"
                      >
                        Send to Clients
                      </button>
                    )}
                  </div>

                  {/* Send Modal */}
                  {selectedDoc === doc.id && (
                    <div className="mt-4 p-4 bg-gray-50 rounded-lg">
                      <h5 className="font-medium mb-2">Send to all parties</h5>
                      <textarea
                        value={message}
                        onChange={e => setMessage(e.target.value)}
                        placeholder="Optional message to include..."
                        className="input mb-3"
                        rows={3}
                      />
                      <div className="flex items-center space-x-2">
                        <button
                          onClick={() => handleSendDocument(doc.id)}
                          disabled={isSending === doc.id}
                          className="btn-primary"
                        >
                          {isSending === doc.id ? 'Sending...' : 'Send Now'}
                        </button>
                        <button
                          onClick={() => {
                            setSelectedDoc(null);
                            setMessage('');
                          }}
                          className="btn-secondary"
                        >
                          Cancel
                        </button>
                      </div>
                    </div>
                  )}
                </div>
              );
            })}
          </div>
        )}
      </div>
    </div>
  );
}
