'use client';

import { useState, useEffect } from 'react';
import { testHubSpotConnectionAction, getHubSpotStatusAction } from '@/modules/integration/hubspot/actions';

export function HubSpotIntegrationCard() {
  const [status, setStatus] = useState<any>(null);
  const [testing, setTesting] = useState(false);
  const [testResult, setTestResult] = useState<any>(null);

  useEffect(() => {
    loadStatus();
  }, []);

  const loadStatus = async () => {
    const result = await getHubSpotStatusAction();
    if (result.success) {
      setStatus(result.data);
    }
  };

  const handleTest = async () => {
    setTesting(true);
    setTestResult(null);
    const result = await testHubSpotConnectionAction();
    setTestResult(result);
    setTesting(false);
  };

  return (
    <div className="card">
      <div className="flex items-start justify-between mb-6">
        <div className="flex items-center space-x-3">
          <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
            <svg className="w-6 h-6 text-orange-600" fill="currentColor" viewBox="0 0 24 24">
              <path d="M18.5 2h-13C4.12 2 3 3.12 3 4.5v15C3 20.88 4.12 22 5.5 22h13c1.38 0 2.5-1.12 2.5-2.5v-15C21 3.12 19.88 2 18.5 2zM12 17c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5zm0-8c-1.65 0-3 1.35-3 3s1.35 3 3 3 3-1.35 3-3-1.35-3-3-3z"/>
            </svg>
          </div>
          <div>
            <h2 className="text-xl font-semibold">HubSpot CRM</h2>
            <p className="text-sm text-gray-600">Sync contacts and deals automatically</p>
          </div>
        </div>
        
        {status && (
          <div className={`px-3 py-1 rounded-full text-sm font-medium ${
            status.enabled ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-600'
          }`}>
            {status.enabled ? 'Active' : 'Inactive'}
          </div>
        )}
      </div>

      {/* Configuration Status */}
      <div className="mb-6 space-y-3">
        <div className="flex items-center justify-between p-3 bg-gray-50 rounded">
          <span className="text-sm font-medium text-gray-700">Integration Enabled</span>
          <span className={`text-sm ${status?.enabled ? 'text-green-600' : 'text-gray-500'}`}>
            {status?.enabled ? '✓ Yes' : '✗ No'}
          </span>
        </div>
        
        <div className="flex items-center justify-between p-3 bg-gray-50 rounded">
          <span className="text-sm font-medium text-gray-700">API Key Configured</span>
          <span className={`text-sm ${status?.hasApiKey ? 'text-green-600' : 'text-gray-500'}`}>
            {status?.hasApiKey ? '✓ Yes' : '✗ No'}
          </span>
        </div>
      </div>

      {/* Features */}
      <div className="mb-6">
        <h3 className="font-semibold mb-3">Features</h3>
        <ul className="space-y-2 text-sm text-gray-700">
          <li className="flex items-center">
            <svg className="w-4 h-4 text-green-600 mr-2" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
            </svg>
            Auto-create contacts on enquiry
          </li>
          <li className="flex items-center">
            <svg className="w-4 h-4 text-green-600 mr-2" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
            </svg>
            Create deals for each case
          </li>
          <li className="flex items-center">
            <svg className="w-4 h-4 text-green-600 mr-2" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
            </svg>
            Update stages on deposit/balance paid
          </li>
          <li className="flex items-center">
            <svg className="w-4 h-4 text-green-600 mr-2" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
            </svg>
            Sync deal amounts automatically
          </li>
          <li className="flex items-center">
            <svg className="w-4 h-4 text-green-600 mr-2" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
            </svg>
            Non-blocking (failures don't break system)
          </li>
        </ul>
      </div>

      {/* Test Connection */}
      <div className="border-t pt-4">
        <button
          onClick={handleTest}
          disabled={testing || !status?.enabled}
          className="btn-primary w-full"
        >
          {testing ? 'Testing Connection...' : 'Test Connection'}
        </button>

        {testResult && (
          <div className={`mt-4 p-3 rounded ${
            testResult.success && testResult.data?.connected
              ? 'bg-green-50 border border-green-200 text-green-800'
              : 'bg-yellow-50 border border-yellow-200 text-yellow-800'
          }`}>
            <p className="text-sm">
              {testResult.data?.message || testResult.error}
            </p>
          </div>
        )}
      </div>

      {/* Setup Instructions */}
      {!status?.enabled && (
        <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded">
          <h4 className="font-semibold text-blue-900 mb-2">Setup Instructions</h4>
          <ol className="text-sm text-blue-800 space-y-2 list-decimal list-inside">
            <li>Get your HubSpot API key from Settings → Integrations → Private Apps</li>
            <li>Add to .env.local: <code className="bg-blue-100 px-1 rounded">HUBSPOT_API_KEY=your_key_here</code></li>
            <li>Enable integration: <code className="bg-blue-100 px-1 rounded">HUBSPOT_ENABLED=true</code></li>
            <li>Restart the application</li>
            <li>Click "Test Connection" to verify</li>
          </ol>
        </div>
      )}
    </div>
  );
}
