'use client';

import { useState } from 'react';
import { updateCertificationsAction, updateLanguagesAction } from '@/modules/mediator/profile/actions';
import { COMMON_CERTIFICATIONS, COMMON_LANGUAGES } from '@/modules/mediator/profile/validation';

export function CertificationsForm({ certifications: initial }: { certifications: string[] }) {
  const [isEditing, setIsEditing] = useState(false);
  const [certifications, setCertifications] = useState(initial);
  const [customInput, setCustomInput] = useState('');
  const [isSaving, setIsSaving] = useState(false);

  const handleSave = async () => {
    setIsSaving(true);
    await updateCertificationsAction(certifications);
    setIsEditing(false);
    setIsSaving(false);
  };

  const toggle = (cert: string) => {
    setCertifications(prev =>
      prev.includes(cert) ? prev.filter(c => c !== cert) : [...prev, cert]
    );
  };

  const addCustom = () => {
    if (customInput.trim() && !certifications.includes(customInput.trim())) {
      setCertifications([...certifications, customInput.trim()]);
      setCustomInput('');
    }
  };

  return (
    <div className="card">
      <div className="flex items-center justify-between mb-4">
        <h2 className="text-xl font-semibold">Certifications</h2>
        {!isEditing && (
          <button onClick={() => setIsEditing(true)} className="text-sm text-primary-600 hover:text-primary-700">Edit</button>
        )}
      </div>

      {isEditing ? (
        <div className="space-y-4">
          <div className="grid grid-cols-2 gap-2">
            {COMMON_CERTIFICATIONS.map(cert => (
              <label key={cert} className="flex items-center p-2 border rounded hover:bg-gray-50 cursor-pointer">
                <input type="checkbox" checked={certifications.includes(cert)} onChange={() => toggle(cert)} className="mr-2" />
                <span className="text-sm">{cert}</span>
              </label>
            ))}
          </div>
          <div className="flex items-center space-x-2">
            <input value={customInput} onChange={e => setCustomInput(e.target.value)} onKeyPress={e => e.key === 'Enter' && (e.preventDefault(), addCustom())} placeholder="Add custom certification" className="input flex-1" />
            <button onClick={addCustom} className="btn-secondary">Add</button>
          </div>
          <div className="flex items-center space-x-3 pt-4">
            <button onClick={handleSave} disabled={isSaving} className="btn-primary">{isSaving ? 'Saving...' : 'Save'}</button>
            <button onClick={() => { setIsEditing(false); setCertifications(initial); }} className="btn-secondary">Cancel</button>
          </div>
        </div>
      ) : (
        <ul className="list-disc list-inside space-y-1">
          {certifications.length > 0 ? certifications.map(cert => <li key={cert}>{cert}</li>) : <p className="text-gray-500">No certifications added</p>}
        </ul>
      )}
    </div>
  );
}

export function LanguagesForm({ languages: initial }: { languages: string[] }) {
  const [isEditing, setIsEditing] = useState(false);
  const [languages, setLanguages] = useState(initial);
  const [customInput, setCustomInput] = useState('');
  const [isSaving, setIsSaving] = useState(false);

  const handleSave = async () => {
    setIsSaving(true);
    await updateLanguagesAction(languages);
    setIsEditing(false);
    setIsSaving(false);
  };

  const toggle = (lang: string) => {
    setLanguages(prev =>
      prev.includes(lang) ? prev.filter(l => l !== lang) : [...prev, lang]
    );
  };

  const addCustom = () => {
    if (customInput.trim() && !languages.includes(customInput.trim())) {
      setLanguages([...languages, customInput.trim()]);
      setCustomInput('');
    }
  };

  return (
    <div className="card">
      <div className="flex items-center justify-between mb-4">
        <h2 className="text-xl font-semibold">Languages</h2>
        {!isEditing && (
          <button onClick={() => setIsEditing(true)} className="text-sm text-primary-600 hover:text-primary-700">Edit</button>
        )}
      </div>

      {isEditing ? (
        <div className="space-y-4">
          <div className="grid grid-cols-3 gap-2">
            {COMMON_LANGUAGES.map(lang => (
              <label key={lang} className="flex items-center p-2 border rounded hover:bg-gray-50 cursor-pointer">
                <input type="checkbox" checked={languages.includes(lang)} onChange={() => toggle(lang)} className="mr-2" />
                <span className="text-sm">{lang}</span>
              </label>
            ))}
          </div>
          <div className="flex items-center space-x-2">
            <input value={customInput} onChange={e => setCustomInput(e.target.value)} onKeyPress={e => e.key === 'Enter' && (e.preventDefault(), addCustom())} placeholder="Add custom language" className="input flex-1" />
            <button onClick={addCustom} className="btn-secondary">Add</button>
          </div>
          <div className="flex items-center space-x-3 pt-4">
            <button onClick={handleSave} disabled={isSaving} className="btn-primary">{isSaving ? 'Saving...' : 'Save'}</button>
            <button onClick={() => { setIsEditing(false); setLanguages(initial); }} className="btn-secondary">Cancel</button>
          </div>
        </div>
      ) : (
        <div className="flex flex-wrap gap-2">
          {languages.length > 0 ? languages.map(lang => <span key={lang} className="px-3 py-1 bg-gray-100 text-gray-800 rounded-full text-sm">{lang}</span>) : <p className="text-gray-500">No languages added</p>}
        </div>
      )}
    </div>
  );
}
