'use client';

import { MediatorPricing } from '@prisma/client';
import { updatePricingAction } from '@/modules/mediator/pricing/actions';
import { useState } from 'react';

interface PricingFormProps {
  pricing: MediatorPricing | null;
  isGstRegistered: boolean;
}

export function PricingForm({ pricing, isGstRegistered }: PricingFormProps) {
  const [isSaving, setIsSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  const handleSubmit = async (formData: FormData) => {
    setIsSaving(true);
    setError(null);
    setSuccess(false);

    const result = await updatePricingAction(formData);

    if (result.error) {
      setError(result.error);
    } else {
      setSuccess(true);
      setTimeout(() => setSuccess(false), 3000);
    }

    setIsSaving(false);
  };

  return (
    <div className="card">
      {success && (
        <div className="mb-4 p-3 bg-green-50 border border-green-200 rounded-lg text-sm text-green-800">
          Pricing updated successfully
        </div>
      )}

      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
          {error}
        </div>
      )}

      <form action={handleSubmit}>
        <div className="space-y-6">
          {/* Pre-mediation Fee */}
          <div>
            <h3 className="font-semibold mb-4">Pre-Mediation Fee</h3>
            <div>
              <label htmlFor="preMediationFee" className="label">
                Fee Amount (optional)
              </label>
              <div className="relative">
                <span className="absolute left-3 top-2 text-gray-500">$</span>
                <input
                  type="number"
                  id="preMediationFee"
                  name="preMediationFee"
                  defaultValue={pricing?.preMediationFee?.toString() || ''}
                  min="0"
                  step="0.01"
                  className="input pl-8"
                  placeholder="0.00"
                />
              </div>
              <p className="text-xs text-gray-500 mt-1">
                Optional flat fee charged before mediation begins
              </p>
            </div>
          </div>

          {/* Hourly Rate */}
          <div className="border-t pt-6">
            <h3 className="font-semibold mb-4">Hourly Rate *</h3>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label htmlFor="hourlyRate" className="label">
                  Rate per Hour
                </label>
                <div className="relative">
                  <span className="absolute left-3 top-2 text-gray-500">$</span>
                  <input
                    type="number"
                    id="hourlyRate"
                    name="hourlyRate"
                    defaultValue={pricing?.hourlyRate?.toString() || ''}
                    required
                    min="1"
                    step="0.01"
                    className="input pl-8"
                    placeholder="200.00"
                  />
                </div>
              </div>

              <div>
                <label htmlFor="hourlyRateBasis" className="label">
                  Rate Basis
                </label>
                <select
                  id="hourlyRateBasis"
                  name="hourlyRateBasis"
                  defaultValue={pricing?.hourlyRateBasis || 'PER_PARTY'}
                  required
                  className="input"
                >
                  <option value="PER_PARTY">Per Party</option>
                  <option value="TOTAL">Total (all parties)</option>
                </select>
                <p className="text-xs text-gray-500 mt-1">
                  Per Party: $200/hr × 2 parties = $400/hr
                </p>
              </div>
            </div>

            <div className="mt-4">
              <label htmlFor="minimumHours" className="label">
                Minimum Hours (optional)
              </label>
              <input
                type="number"
                id="minimumHours"
                name="minimumHours"
                defaultValue={pricing?.minimumHours?.toString() || ''}
                min="0"
                step="0.5"
                className="input w-48"
                placeholder="2.0"
              />
            </div>
          </div>

          {/* Travel */}
          <div className="border-t pt-6">
            <h3 className="font-semibold mb-4">Travel Policy</h3>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label htmlFor="travelIncludedKm" className="label">
                  Included Kilometers
                </label>
                <input
                  type="number"
                  id="travelIncludedKm"
                  name="travelIncludedKm"
                  defaultValue={pricing?.travelIncludedKm?.toString() || ''}
                  min="0"
                  className="input"
                  placeholder="50"
                />
                <p className="text-xs text-gray-500 mt-1">
                  Travel within this distance is free
                </p>
              </div>

              <div>
                <label htmlFor="travelRatePerKm" className="label">
                  Rate per km (beyond included)
                </label>
                <div className="relative">
                  <span className="absolute left-3 top-2 text-gray-500">$</span>
                  <input
                    type="number"
                    id="travelRatePerKm"
                    name="travelRatePerKm"
                    defaultValue={pricing?.travelRatePerKm?.toString() || ''}
                    min="0"
                    step="0.01"
                    className="input pl-8"
                    placeholder="1.50"
                  />
                </div>
              </div>
            </div>
          </div>

          {/* Accommodation */}
          <div className="border-t pt-6">
            <h3 className="font-semibold mb-4">Accommodation Policy</h3>
            <div className="space-y-4">
              <div>
                <label htmlFor="accommodationPolicy" className="label">
                  Policy Description
                </label>
                <textarea
                  id="accommodationPolicy"
                  name="accommodationPolicy"
                  defaultValue={pricing?.accommodationPolicy || ''}
                  className="input"
                  rows={2}
                  maxLength={500}
                  placeholder="e.g., Accommodation costs apply for locations requiring overnight stay..."
                />
              </div>

              <div>
                <label htmlFor="accommodationRate" className="label">
                  Rate (if applicable)
                </label>
                <div className="relative w-64">
                  <span className="absolute left-3 top-2 text-gray-500">$</span>
                  <input
                    type="number"
                    id="accommodationRate"
                    name="accommodationRate"
                    defaultValue={pricing?.accommodationRate?.toString() || ''}
                    min="0"
                    step="0.01"
                    className="input pl-8"
                    placeholder="0.00"
                  />
                </div>
              </div>
            </div>
          </div>

          {/* Extra Costs */}
          <div className="border-t pt-6">
            <h3 className="font-semibold mb-4">Additional Costs</h3>
            <div>
              <label htmlFor="extraCostsNotes" className="label">
                Notes (optional)
              </label>
              <textarea
                id="extraCostsNotes"
                name="extraCostsNotes"
                defaultValue={pricing?.extraCostsNotes || ''}
                className="input"
                rows={3}
                maxLength={1000}
                placeholder="Any additional costs or fees clients should be aware of..."
              />
            </div>
          </div>

          {/* Submit */}
          <div className="border-t pt-6">
            <button
              type="submit"
              disabled={isSaving}
              className="btn-primary"
            >
              {isSaving ? 'Saving...' : 'Save Pricing'}
            </button>
          </div>
        </div>
      </form>
    </div>
  );
}
