'use client';

import { useState } from 'react';
import { updateProfileAction } from '@/modules/mediator/profile/actions';

interface ProfileBasicInfoProps {
  mediator: any;
  userId: string;
  disabled?: boolean;
}

export function ProfileBasicInfo({ mediator, userId, disabled }: ProfileBasicInfoProps) {
  const [saving, setSaving] = useState(false);
  const [message, setMessage] = useState<{ type: 'success' | 'error'; text: string } | null>(null);

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setSaving(true);
    setMessage(null);

    const formData = new FormData(e.currentTarget);
    const result = await updateProfileAction(mediator.id, userId, formData);

    if (result.error) {
      setMessage({ type: 'error', text: result.error });
    } else {
      setMessage({ type: 'success', text: 'Profile updated successfully' });
    }

    setSaving(false);
  };

  return (
    <div id="basic" className="card">
      <h2 className="text-xl font-semibold mb-6">Basic Information</h2>

      <form onSubmit={handleSubmit}>
        <div className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <label htmlFor="firstName" className="label">
                First Name *
              </label>
              <input
                id="firstName"
                name="firstName"
                type="text"
                defaultValue={mediator.firstName}
                required
                disabled={disabled}
                className="input"
              />
            </div>

            <div>
              <label htmlFor="lastName" className="label">
                Last Name *
              </label>
              <input
                id="lastName"
                name="lastName"
                type="text"
                defaultValue={mediator.lastName}
                required
                disabled={disabled}
                className="input"
              />
            </div>
          </div>

          <div>
            <label htmlFor="displayName" className="label">
              Display Name (optional)
            </label>
            <input
              id="displayName"
              name="displayName"
              type="text"
              defaultValue={mediator.displayName || ''}
              placeholder="e.g., John Smith, Esq."
              disabled={disabled}
              className="input"
            />
            <p className="text-xs text-gray-500 mt-1">
              How you'd like your name to appear on your practice page
            </p>
          </div>

          <div>
            <label htmlFor="bio" className="label">
              Bio *
            </label>
            <textarea
              id="bio"
              name="bio"
              rows={6}
              defaultValue={mediator.bio || ''}
              required
              disabled={disabled}
              className="input"
              placeholder="Tell clients about your experience, approach to mediation, and areas of expertise..."
            />
            <p className="text-xs text-gray-500 mt-1">
              Minimum 50 characters. This will be displayed on your practice page.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <label htmlFor="phone" className="label">
                Phone Number *
              </label>
              <input
                id="phone"
                name="phone"
                type="tel"
                defaultValue={mediator.phone || ''}
                required
                disabled={disabled}
                className="input"
                placeholder="+61 4XX XXX XXX"
              />
            </div>

            <div>
              <label htmlFor="yearsExperience" className="label">
                Years of Experience
              </label>
              <input
                id="yearsExperience"
                name="yearsExperience"
                type="number"
                min="0"
                max="100"
                defaultValue={mediator.yearsExperience || ''}
                disabled={disabled}
                className="input"
              />
            </div>
          </div>

          <div>
            <label htmlFor="officeAddress" className="label">
              Office Address (optional)
            </label>
            <textarea
              id="officeAddress"
              name="officeAddress"
              rows={2}
              defaultValue={mediator.officeAddress || ''}
              disabled={disabled}
              className="input"
              placeholder="123 Main St, Sydney NSW 2000"
            />
          </div>

          {message && (
            <div
              className={`p-4 rounded-lg ${
                message.type === 'success'
                  ? 'bg-green-50 text-green-800 border border-green-200'
                  : 'bg-red-50 text-red-800 border border-red-200'
              }`}
            >
              {message.text}
            </div>
          )}

          <div className="flex justify-end">
            <button
              type="submit"
              disabled={saving || disabled}
              className="btn-primary"
            >
              {saving ? 'Saving...' : 'Save Changes'}
            </button>
          </div>
        </div>
      </form>
    </div>
  );
}
