'use client';

import { useState } from 'react';
import { submitForApprovalAction } from '@/modules/mediator/profile/actions';

interface ProfileCompletionProps {
  percentage: number;
  missingFields: string[];
  status: string;
  mediatorId: string;
  userId: string;
}

export function ProfileCompletion({
  percentage,
  missingFields,
  status,
  mediatorId,
  userId,
}: ProfileCompletionProps) {
  const [submitting, setSubmitting] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const isComplete = percentage === 100;
  const canSubmit = isComplete && status === 'PENDING_APPROVAL';
  const isApproved = status === 'APPROVED' || status === 'PUBLISHED';
  const isRejected = status === 'REJECTED';

  const handleSubmit = async () => {
    setSubmitting(true);
    setError(null);

    const result = await submitForApprovalAction(mediatorId, userId);

    if (result.error) {
      setError(result.error);
      setSubmitting(false);
    } else {
      // Success - page will revalidate
      setSubmitting(false);
    }
  };

  return (
    <div className="card mb-8">
      <div className="flex items-start justify-between mb-4">
        <div className="flex-1">
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            Profile Completion
          </h3>
          <div className="w-full bg-gray-200 rounded-full h-2.5">
            <div
              className="bg-primary-600 h-2.5 rounded-full transition-all"
              style={{ width: `${percentage}%` }}
            />
          </div>
          <p className="text-sm text-gray-600 mt-2">
            {percentage}% complete
          </p>
        </div>

        {canSubmit && (
          <button
            onClick={handleSubmit}
            disabled={submitting}
            className="btn-primary ml-4"
          >
            {submitting ? 'Submitting...' : 'Submit for Approval'}
          </button>
        )}

        {isApproved && (
          <div className="ml-4 px-4 py-2 bg-green-100 text-green-800 rounded-lg font-medium">
            ✓ Approved
          </div>
        )}
      </div>

      {error && (
        <div className="mt-4 bg-red-50 border border-red-200 rounded-lg p-3 text-sm text-red-800">
          {error}
        </div>
      )}

      {missingFields.length > 0 && (
        <div className="mt-4 bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="font-medium text-yellow-900 mb-2">
            Missing required fields:
          </div>
          <ul className="text-sm text-yellow-800 space-y-1">
            {missingFields.map((field, i) => (
              <li key={i}>• {field}</li>
            ))}
          </ul>
        </div>
      )}

      {isRejected && (
        <div className="mt-4 bg-red-50 border border-red-200 rounded-lg p-4">
          <div className="font-medium text-red-900 mb-2">
            Profile Rejected
          </div>
          <p className="text-sm text-red-800">
            Your profile was not approved. Please review and update your information, then submit again.
          </p>
        </div>
      )}

      {isComplete && !isApproved && !isRejected && status !== 'PENDING_APPROVAL' && (
        <div className="mt-4 bg-blue-50 border border-blue-200 rounded-lg p-4">
          <p className="text-sm text-blue-900">
            Your profile is complete! Click "Submit for Approval" to get reviewed by our team.
          </p>
        </div>
      )}
    </div>
  );
}
