'use client';

import { Mediator } from '@prisma/client';
import { updateProfileAction } from '@/modules/mediator/profile/actions';
import { useState } from 'react';

interface ProfileFormProps {
  mediator: Mediator;
}

export function ProfileForm({ mediator }: ProfileFormProps) {
  const [isEditing, setIsEditing] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  const handleSubmit = async (formData: FormData) => {
    setIsSaving(true);
    setError(null);
    setSuccess(false);

    const result = await updateProfileAction(formData);

    if (result.error) {
      setError(result.error);
    } else {
      setSuccess(true);
      setIsEditing(false);
      setTimeout(() => setSuccess(false), 3000);
    }

    setIsSaving(false);
  };

  return (
    <div className="card">
      <div className="flex items-center justify-between mb-4">
        <h2 className="text-xl font-semibold">Basic Information</h2>
        {!isEditing && (
          <button
            onClick={() => setIsEditing(true)}
            className="text-sm text-primary-600 hover:text-primary-700"
          >
            Edit
          </button>
        )}
      </div>

      {success && (
        <div className="mb-4 p-3 bg-green-50 border border-green-200 rounded-lg text-sm text-green-800">
          Profile updated successfully
        </div>
      )}

      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
          {error}
        </div>
      )}

      {isEditing ? (
        <form action={handleSubmit}>
          <div className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label htmlFor="firstName" className="label">
                  First Name *
                </label>
                <input
                  type="text"
                  id="firstName"
                  name="firstName"
                  defaultValue={mediator.firstName}
                  required
                  className="input"
                />
              </div>

              <div>
                <label htmlFor="lastName" className="label">
                  Last Name *
                </label>
                <input
                  type="text"
                  id="lastName"
                  name="lastName"
                  defaultValue={mediator.lastName}
                  required
                  className="input"
                />
              </div>
            </div>

            <div>
              <label htmlFor="displayName" className="label">
                Display Name (optional)
              </label>
              <input
                type="text"
                id="displayName"
                name="displayName"
                defaultValue={mediator.displayName || ''}
                className="input"
                placeholder="e.g., John Smith, Esq."
              />
              <p className="text-xs text-gray-500 mt-1">
                How your name appears on your public profile
              </p>
            </div>

            <div>
              <label htmlFor="phone" className="label">
                Phone Number *
              </label>
              <input
                type="tel"
                id="phone"
                name="phone"
                defaultValue={mediator.phone || ''}
                required
                className="input"
                placeholder="+61 4XX XXX XXX"
              />
            </div>

            <div>
              <label htmlFor="officeAddress" className="label">
                Office Address (optional)
              </label>
              <textarea
                id="officeAddress"
                name="officeAddress"
                defaultValue={mediator.officeAddress || ''}
                className="input"
                rows={2}
                placeholder="123 Main St, Sydney NSW 2000"
              />
            </div>

            <div>
              <label htmlFor="yearsExperience" className="label">
                Years of Experience *
              </label>
              <input
                type="number"
                id="yearsExperience"
                name="yearsExperience"
                defaultValue={mediator.yearsExperience || ''}
                required
                min="0"
                max="70"
                className="input"
              />
            </div>

            <div>
              <label htmlFor="bio" className="label">
                Bio *
              </label>
              <textarea
                id="bio"
                name="bio"
                defaultValue={mediator.bio || ''}
                required
                className="input"
                rows={6}
                minLength={50}
                maxLength={2000}
                placeholder="Tell clients about your experience, approach to mediation, and what makes you unique..."
              />
              <p className="text-xs text-gray-500 mt-1">
                Minimum 50 characters, maximum 2000 characters
              </p>
            </div>

            <div className="flex items-center space-x-3 pt-4">
              <button
                type="submit"
                disabled={isSaving}
                className="btn-primary"
              >
                {isSaving ? 'Saving...' : 'Save Changes'}
              </button>
              <button
                type="button"
                onClick={() => {
                  setIsEditing(false);
                  setError(null);
                }}
                className="btn-secondary"
              >
                Cancel
              </button>
            </div>
          </div>
        </form>
      ) : (
        <div className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <div className="text-sm font-medium text-gray-600">Name</div>
              <div className="text-base">
                {mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
              </div>
            </div>

            <div>
              <div className="text-sm font-medium text-gray-600">Phone</div>
              <div className="text-base">{mediator.phone || 'Not set'}</div>
            </div>
          </div>

          {mediator.officeAddress && (
            <div>
              <div className="text-sm font-medium text-gray-600">Office Address</div>
              <div className="text-base">{mediator.officeAddress}</div>
            </div>
          )}

          <div>
            <div className="text-sm font-medium text-gray-600">Years of Experience</div>
            <div className="text-base">{mediator.yearsExperience || 'Not set'}</div>
          </div>

          {mediator.bio && (
            <div>
              <div className="text-sm font-medium text-gray-600">Bio</div>
              <div className="text-base whitespace-pre-line">{mediator.bio}</div>
            </div>
          )}
        </div>
      )}
    </div>
  );
}
