'use client';

import { useState } from 'react';
import { updatePricingAction } from '@/modules/mediator/pricing/actions';

interface ProfilePricingProps {
  mediator: any;
  pricing: any;
  userId: string;
  disabled?: boolean;
}

export function ProfilePricing({ mediator, pricing, userId, disabled }: ProfilePricingProps) {
  const [saving, setSaving] = useState(false);
  const [message, setMessage] = useState<{ type: 'success' | 'error'; text: string } | null>(null);

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setSaving(true);
    setMessage(null);

    const formData = new FormData(e.currentTarget);
    const result = await updatePricingAction(mediator.id, userId, formData);

    if (result.error) {
      setMessage({ type: 'error', text: result.error });
    } else {
      setMessage({ type: 'success', text: 'Pricing updated successfully' });
    }

    setSaving(false);
  };

  return (
    <div id="pricing" className="card">
      <h2 className="text-xl font-semibold mb-6">Pricing Configuration</h2>

      <form onSubmit={handleSubmit}>
        <div className="space-y-6">
          {/* Pre-mediation Fee */}
          <div>
            <label htmlFor="preMediationFee" className="label">
              Pre-Mediation Fee (Optional)
            </label>
            <div className="relative">
              <span className="absolute left-3 top-2.5 text-gray-500">$</span>
              <input
                id="preMediationFee"
                name="preMediationFee"
                type="number"
                step="0.01"
                min="0"
                defaultValue={pricing.preMediationFee || ''}
                disabled={disabled}
                className="input pl-7"
                placeholder="0.00"
              />
            </div>
            <p className="text-xs text-gray-500 mt-1">
              Optional flat fee charged before mediation sessions begin
            </p>
          </div>

          {/* Hourly Rate */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <label htmlFor="hourlyRate" className="label">
                Hourly Rate *
              </label>
              <div className="relative">
                <span className="absolute left-3 top-2.5 text-gray-500">$</span>
                <input
                  id="hourlyRate"
                  name="hourlyRate"
                  type="number"
                  step="0.01"
                  min="1"
                  defaultValue={pricing.hourlyRate || ''}
                  required
                  disabled={disabled}
                  className="input pl-7"
                  placeholder="200.00"
                />
              </div>
              {mediator.isGstRegistered && (
                <p className="text-xs text-gray-500 mt-1">
                  GST will be added to this rate
                </p>
              )}
            </div>

            <div>
              <label htmlFor="hourlyRateBasis" className="label">
                Rate Basis *
              </label>
              <select
                id="hourlyRateBasis"
                name="hourlyRateBasis"
                defaultValue={pricing.hourlyRateBasis || 'PER_PARTY'}
                required
                disabled={disabled}
                className="input"
              >
                <option value="PER_PARTY">Per Party</option>
                <option value="TOTAL">Total (All Parties)</option>
              </select>
              <p className="text-xs text-gray-500 mt-1">
                {pricing.hourlyRateBasis === 'PER_PARTY'
                  ? 'Rate multiplied by number of parties'
                  : 'Fixed rate regardless of party count'}
              </p>
            </div>
          </div>

          <div>
            <label htmlFor="minimumHours" className="label">
              Minimum Hours (Optional)
            </label>
            <input
              id="minimumHours"
              name="minimumHours"
              type="number"
              step="0.5"
              min="0"
              defaultValue={pricing.minimumHours || ''}
              disabled={disabled}
              className="input"
              placeholder="2.0"
            />
            <p className="text-xs text-gray-500 mt-1">
              Minimum billable hours per session
            </p>
          </div>

          {/* Travel Policy */}
          <div className="border-t pt-6">
            <h3 className="font-semibold text-gray-900 mb-4">Travel Policy</h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label htmlFor="travelIncludedKm" className="label">
                  Included Kilometres
                </label>
                <input
                  id="travelIncludedKm"
                  name="travelIncludedKm"
                  type="number"
                  min="0"
                  defaultValue={pricing.travelIncludedKm || ''}
                  disabled={disabled}
                  className="input"
                  placeholder="50"
                />
                <p className="text-xs text-gray-500 mt-1">
                  Free travel distance (km)
                </p>
              </div>

              <div>
                <label htmlFor="travelRatePerKm" className="label">
                  Rate per Additional Kilometre
                </label>
                <div className="relative">
                  <span className="absolute left-3 top-2.5 text-gray-500">$</span>
                  <input
                    id="travelRatePerKm"
                    name="travelRatePerKm"
                    type="number"
                    step="0.01"
                    min="0"
                    defaultValue={pricing.travelRatePerKm || ''}
                    disabled={disabled}
                    className="input pl-7"
                    placeholder="1.50"
                  />
                </div>
              </div>
            </div>
          </div>

          {/* Accommodation */}
          <div className="border-t pt-6">
            <h3 className="font-semibold text-gray-900 mb-4">Accommodation Policy</h3>
            
            <div className="space-y-4">
              <div>
                <label htmlFor="accommodationPolicy" className="label">
                  Policy Description
                </label>
                <textarea
                  id="accommodationPolicy"
                  name="accommodationPolicy"
                  rows={3}
                  defaultValue={pricing.accommodationPolicy || ''}
                  disabled={disabled}
                  className="input"
                  placeholder="Describe your accommodation policy, e.g., 'Accommodation charged at cost for sessions requiring overnight stay'"
                />
              </div>

              <div>
                <label htmlFor="accommodationRate" className="label">
                  Accommodation Rate
                </label>
                <div className="relative">
                  <span className="absolute left-3 top-2.5 text-gray-500">$</span>
                  <input
                    id="accommodationRate"
                    name="accommodationRate"
                    type="number"
                    step="0.01"
                    min="0"
                    defaultValue={pricing.accommodationRate || ''}
                    disabled={disabled}
                    className="input pl-7"
                    placeholder="200.00"
                  />
                </div>
                <p className="text-xs text-gray-500 mt-1">
                  Per night or flat rate
                </p>
              </div>
            </div>
          </div>

          {/* Extra Costs */}
          <div className="border-t pt-6">
            <h3 className="font-semibold text-gray-900 mb-4">Additional Costs</h3>
            
            <div>
              <label htmlFor="extraCostsNotes" className="label">
                Extra Costs Notes
              </label>
              <textarea
                id="extraCostsNotes"
                name="extraCostsNotes"
                rows={3}
                defaultValue={pricing.extraCostsNotes || ''}
                disabled={disabled}
                className="input"
                placeholder="Any other fees or costs that may apply, e.g., venue hire, document preparation, etc."
              />
            </div>
          </div>

          {message && (
            <div
              className={`p-4 rounded-lg ${
                message.type === 'success'
                  ? 'bg-green-50 text-green-800 border border-green-200'
                  : 'bg-red-50 text-red-800 border border-red-200'
              }`}
            >
              {message.text}
            </div>
          )}

          <div className="flex justify-end pt-6 border-t">
            <button
              type="submit"
              disabled={saving || disabled}
              className="btn-primary"
            >
              {saving ? 'Saving...' : 'Save Pricing'}
            </button>
          </div>
        </div>
      </form>
    </div>
  );
}
