'use client';

import { useState } from 'react';
import { 
  updateSpecializationsAction, 
  updateCertificationsAction,
  updateLanguagesAction 
} from '@/modules/mediator/profile/actions';
import { 
  SPECIALIZATIONS, 
  CERTIFICATIONS, 
  LANGUAGES 
} from '@/modules/mediator/profile/validation';

interface ProfileSpecializationsProps {
  mediator: any;
  userId: string;
  disabled?: boolean;
}

export function ProfileSpecializations({ mediator, userId, disabled }: ProfileSpecializationsProps) {
  const [selectedSpecs, setSelectedSpecs] = useState<string[]>(mediator.specializations || []);
  const [selectedCerts, setSelectedCerts] = useState<string[]>(mediator.certifications || []);
  const [selectedLangs, setSelectedLangs] = useState<string[]>(mediator.languages || []);
  
  const [saving, setSaving] = useState<'specs' | 'certs' | 'langs' | null>(null);
  const [message, setMessage] = useState<{ type: 'success' | 'error'; text: string } | null>(null);

  const handleSaveSpecializations = async () => {
    setSaving('specs');
    setMessage(null);

    const result = await updateSpecializationsAction(mediator.id, userId, selectedSpecs);

    if (result.error) {
      setMessage({ type: 'error', text: result.error });
    } else {
      setMessage({ type: 'success', text: 'Specializations updated successfully' });
    }

    setSaving(null);
  };

  const handleSaveCertifications = async () => {
    setSaving('certs');
    setMessage(null);

    const result = await updateCertificationsAction(mediator.id, userId, selectedCerts);

    if (result.error) {
      setMessage({ type: 'error', text: result.error });
    } else {
      setMessage({ type: 'success', text: 'Certifications updated successfully' });
    }

    setSaving(null);
  };

  const handleSaveLanguages = async () => {
    setSaving('langs');
    setMessage(null);

    const result = await updateLanguagesAction(mediator.id, userId, selectedLangs);

    if (result.error) {
      setMessage({ type: 'error', text: result.error });
    } else {
      setMessage({ type: 'success', text: 'Languages updated successfully' });
    }

    setSaving(null);
  };

  const toggleSpec = (spec: string) => {
    setSelectedSpecs(prev =>
      prev.includes(spec) ? prev.filter(s => s !== spec) : [...prev, spec]
    );
  };

  const toggleCert = (cert: string) => {
    setSelectedCerts(prev =>
      prev.includes(cert) ? prev.filter(c => c !== cert) : [...prev, cert]
    );
  };

  const toggleLang = (lang: string) => {
    setSelectedLangs(prev =>
      prev.includes(lang) ? prev.filter(l => l !== lang) : [...prev, lang]
    );
  };

  return (
    <div id="specializations" className="card">
      <h2 className="text-xl font-semibold mb-6">Specializations & Credentials</h2>

      <div className="space-y-8">
        {/* Specializations */}
        <div>
          <div className="flex items-center justify-between mb-4">
            <label className="label">
              Specializations * (Select at least one)
            </label>
            <button
              type="button"
              onClick={handleSaveSpecializations}
              disabled={saving === 'specs' || disabled || selectedSpecs.length === 0}
              className="btn-primary"
            >
              {saving === 'specs' ? 'Saving...' : 'Save Specializations'}
            </button>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
            {SPECIALIZATIONS.map(spec => (
              <label
                key={spec}
                className={`flex items-center p-3 border rounded-lg cursor-pointer transition-colors ${
                  selectedSpecs.includes(spec)
                    ? 'border-primary-500 bg-primary-50'
                    : 'border-gray-300 hover:border-gray-400'
                } ${disabled ? 'opacity-50 cursor-not-allowed' : ''}`}
              >
                <input
                  type="checkbox"
                  checked={selectedSpecs.includes(spec)}
                  onChange={() => toggleSpec(spec)}
                  disabled={disabled}
                  className="h-4 w-4 text-primary-600 focus:ring-primary-500 border-gray-300 rounded"
                />
                <span className="ml-2 text-sm text-gray-900">{spec}</span>
              </label>
            ))}
          </div>
        </div>

        {/* Certifications */}
        <div>
          <div className="flex items-center justify-between mb-4">
            <label className="label">
              Certifications * (Select at least one)
            </label>
            <button
              type="button"
              onClick={handleSaveCertifications}
              disabled={saving === 'certs' || disabled || selectedCerts.length === 0}
              className="btn-primary"
            >
              {saving === 'certs' ? 'Saving...' : 'Save Certifications'}
            </button>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            {CERTIFICATIONS.map(cert => (
              <label
                key={cert}
                className={`flex items-center p-3 border rounded-lg cursor-pointer transition-colors ${
                  selectedCerts.includes(cert)
                    ? 'border-primary-500 bg-primary-50'
                    : 'border-gray-300 hover:border-gray-400'
                } ${disabled ? 'opacity-50 cursor-not-allowed' : ''}`}
              >
                <input
                  type="checkbox"
                  checked={selectedCerts.includes(cert)}
                  onChange={() => toggleCert(cert)}
                  disabled={disabled}
                  className="h-4 w-4 text-primary-600 focus:ring-primary-500 border-gray-300 rounded"
                />
                <span className="ml-2 text-sm text-gray-900">{cert}</span>
              </label>
            ))}
          </div>
        </div>

        {/* Languages */}
        <div>
          <div className="flex items-center justify-between mb-4">
            <label className="label">
              Languages (Optional)
            </label>
            <button
              type="button"
              onClick={handleSaveLanguages}
              disabled={saving === 'langs' || disabled}
              className="btn-primary"
            >
              {saving === 'langs' ? 'Saving...' : 'Save Languages'}
            </button>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
            {LANGUAGES.map(lang => (
              <label
                key={lang}
                className={`flex items-center p-3 border rounded-lg cursor-pointer transition-colors ${
                  selectedLangs.includes(lang)
                    ? 'border-primary-500 bg-primary-50'
                    : 'border-gray-300 hover:border-gray-400'
                } ${disabled ? 'opacity-50 cursor-not-allowed' : ''}`}
              >
                <input
                  type="checkbox"
                  checked={selectedLangs.includes(lang)}
                  onChange={() => toggleLang(lang)}
                  disabled={disabled}
                  className="h-4 w-4 text-primary-600 focus:ring-primary-500 border-gray-300 rounded"
                />
                <span className="ml-2 text-sm text-gray-900">{lang}</span>
              </label>
            ))}
          </div>
        </div>

        {message && (
          <div
            className={`p-4 rounded-lg ${
              message.type === 'success'
                ? 'bg-green-50 text-green-800 border border-green-200'
                : 'bg-red-50 text-red-800 border border-red-200'
            }`}
          >
            {message.text}
          </div>
        )}
      </div>
    </div>
  );
}
