'use client';

import { Mediator } from '@prisma/client';
import { submitForApprovalAction } from '@/modules/mediator/profile/actions';
import { useState } from 'react';

interface ProfileStatusProps {
  mediator: Mediator;
  isComplete: boolean;
}

export function ProfileStatus({ mediator, isComplete }: ProfileStatusProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleSubmit = async () => {
    setIsSubmitting(true);
    setError(null);

    const result = await submitForApprovalAction();
    
    if (result.error) {
      setError(result.error);
    }
    
    setIsSubmitting(false);
  };

  // Pending approval
  if (mediator.status === 'PENDING_APPROVAL') {
    return (
      <div className="mb-6 bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div className="flex items-start">
          <svg className="w-5 h-5 text-yellow-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
          </svg>
          <div className="flex-1">
            <div className="font-medium text-yellow-900">
              Profile Under Review
            </div>
            <p className="text-sm text-yellow-800 mt-1">
              Your profile has been submitted and is being reviewed by our team.
              You'll be notified once it's approved.
            </p>
          </div>
        </div>
      </div>
    );
  }

  // Approved (not published)
  if (mediator.status === 'APPROVED') {
    return (
      <div className="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
        <div className="flex items-start">
          <svg className="w-5 h-5 text-green-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
          </svg>
          <div className="flex-1">
            <div className="font-medium text-green-900">
              Profile Approved
            </div>
            <p className="text-sm text-green-800 mt-1">
              Your profile has been approved! It will be published shortly.
            </p>
          </div>
        </div>
      </div>
    );
  }

  // Published
  if (mediator.status === 'PUBLISHED') {
    return (
      <div className="mb-6 bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex items-start">
          <svg className="w-5 h-5 text-blue-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
            <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
            <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
          </svg>
          <div className="flex-1">
            <div className="font-medium text-blue-900">
              Profile Published
            </div>
            <p className="text-sm text-blue-800 mt-1">
              Your profile is live and visible to potential clients.{' '}
              <a href={`/practices/${mediator.slug}`} target="_blank" className="underline">
                View public page
              </a>
            </p>
          </div>
        </div>
      </div>
    );
  }

  // Rejected
  if (mediator.status === 'REJECTED') {
    return (
      <div className="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
        <div className="flex items-start">
          <svg className="w-5 h-5 text-red-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
          </svg>
          <div className="flex-1">
            <div className="font-medium text-red-900">
              Profile Not Approved
            </div>
            {mediator.approvalNotes && (
              <p className="text-sm text-red-800 mt-1">
                <strong>Reason:</strong> {mediator.approvalNotes}
              </p>
            )}
            <p className="text-sm text-red-800 mt-2">
              Please update your profile and resubmit for approval.
            </p>
          </div>
        </div>
        {isComplete && (
          <div className="mt-4">
            <button
              onClick={handleSubmit}
              disabled={isSubmitting}
              className="btn-primary"
            >
              {isSubmitting ? 'Submitting...' : 'Resubmit for Approval'}
            </button>
            {error && (
              <p className="text-sm text-red-600 mt-2">{error}</p>
            )}
          </div>
        )}
      </div>
    );
  }

  // Suspended
  if (mediator.status === 'SUSPENDED') {
    return (
      <div className="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
        <div className="flex items-start">
          <svg className="w-5 h-5 text-red-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M13.477 14.89A6 6 0 015.11 6.524l8.367 8.368zm1.414-1.414L6.524 5.11a6 6 0 018.367 8.367zM18 10a8 8 0 11-16 0 8 8 0 0116 0z" clipRule="evenodd" />
          </svg>
          <div className="flex-1">
            <div className="font-medium text-red-900">
              Account Suspended
            </div>
            {mediator.approvalNotes && (
              <p className="text-sm text-red-800 mt-1">
                <strong>Reason:</strong> {mediator.approvalNotes}
              </p>
            )}
            <p className="text-sm text-red-800 mt-2">
              Please contact support for assistance.
            </p>
          </div>
        </div>
      </div>
    );
  }

  // Incomplete profile (default state after registration)
  return (
    <div className="mb-6 bg-gray-50 border border-gray-200 rounded-lg p-4">
      <div className="flex items-start">
        <svg className="w-5 h-5 text-gray-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
          <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
        </svg>
        <div className="flex-1">
          <div className="font-medium text-gray-900">
            Complete Your Profile
          </div>
          <p className="text-sm text-gray-700 mt-1">
            Fill in all required information below, then submit your profile for approval.
          </p>
          {!isComplete && (
            <p className="text-sm text-gray-600 mt-2">
              Required: Bio, Phone, Specializations, Years of Experience
            </p>
          )}
        </div>
      </div>
      {isComplete && (
        <div className="mt-4">
          <button
            onClick={handleSubmit}
            disabled={isSubmitting}
            className="btn-primary"
          >
            {isSubmitting ? 'Submitting...' : 'Submit for Approval'}
          </button>
          {error && (
            <p className="text-sm text-red-600 mt-2">{error}</p>
          )}
        </div>
      )}
    </div>
  );
}
