'use client';

import { useState } from 'react';
import { updateSpecializationsAction } from '@/modules/mediator/profile/actions';
import { COMMON_SPECIALIZATIONS } from '@/modules/mediator/profile/validation';

export function SpecializationsForm({ specializations: initial }: { specializations: string[] }) {
  const [isEditing, setIsEditing] = useState(false);
  const [specializations, setSpecializations] = useState(initial);
  const [customInput, setCustomInput] = useState('');
  const [isSaving, setIsSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleSave = async () => {
    setIsSaving(true);
    setError(null);
    const result = await updateSpecializationsAction(specializations);
    if (result.error) setError(result.error);
    else setIsEditing(false);
    setIsSaving(false);
  };

  const toggleSpec = (spec: string) => {
    setSpecializations(prev =>
      prev.includes(spec) ? prev.filter(s => s !== spec) : [...prev, spec]
    );
  };

  const addCustom = () => {
    if (customInput.trim() && !specializations.includes(customInput.trim())) {
      setSpecializations([...specializations, customInput.trim()]);
      setCustomInput('');
    }
  };

  return (
    <div className="card">
      <div className="flex items-center justify-between mb-4">
        <h2 className="text-xl font-semibold">Specializations *</h2>
        {!isEditing && (
          <button onClick={() => setIsEditing(true)} className="text-sm text-primary-600 hover:text-primary-700">
            Edit
          </button>
        )}
      </div>

      {error && <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">{error}</div>}

      {isEditing ? (
        <div className="space-y-4">
          <div className="grid grid-cols-2 gap-2">
            {COMMON_SPECIALIZATIONS.map(spec => (
              <label key={spec} className="flex items-center p-2 border rounded hover:bg-gray-50 cursor-pointer">
                <input
                  type="checkbox"
                  checked={specializations.includes(spec)}
                  onChange={() => toggleSpec(spec)}
                  className="mr-2"
                />
                <span className="text-sm">{spec}</span>
              </label>
            ))}
          </div>
          <div className="flex items-center space-x-2">
            <input
              type="text"
              value={customInput}
              onChange={e => setCustomInput(e.target.value)}
              onKeyPress={e => e.key === 'Enter' && (e.preventDefault(), addCustom())}
              placeholder="Add custom specialization"
              className="input flex-1"
            />
            <button onClick={addCustom} className="btn-secondary">Add</button>
          </div>
          {specializations.filter(s => !COMMON_SPECIALIZATIONS.includes(s as any)).map(spec => (
            <div key={spec} className="flex items-center justify-between p-2 bg-gray-50 rounded">
              <span>{spec}</span>
              <button onClick={() => setSpecializations(specializations.filter(s => s !== spec))} className="text-red-600 text-sm">Remove</button>
            </div>
          ))}
          <div className="flex items-center space-x-3 pt-4">
            <button onClick={handleSave} disabled={isSaving || specializations.length === 0} className="btn-primary">
              {isSaving ? 'Saving...' : 'Save'}
            </button>
            <button onClick={() => { setIsEditing(false); setSpecializations(initial); setError(null); }} className="btn-secondary">
              Cancel
            </button>
          </div>
        </div>
      ) : (
        <div className="flex flex-wrap gap-2">
          {specializations.length > 0 ? (
            specializations.map(spec => (
              <span key={spec} className="px-3 py-1 bg-primary-100 text-primary-800 rounded-full text-sm">
                {spec}
              </span>
            ))
          ) : (
            <p className="text-gray-500">No specializations added</p>
          )}
        </div>
      )}
    </div>
  );
}
