'use client';

import { Mediator } from '@prisma/client';
import { useState } from 'react';
import { startConnectOnboardingAction, checkConnectStatusAction, createDashboardLoginLinkAction } from '@/modules/payment/actions';

interface ConnectOnboardingCardProps {
  mediator: Mediator;
  connectStatus: any;
}

export function ConnectOnboardingCard({ mediator, connectStatus }: ConnectOnboardingCardProps) {
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleStartOnboarding = async () => {
    setIsLoading(true);
    setError(null);

    const result = await startConnectOnboardingAction();

    if (result.error) {
      setError(result.error);
      setIsLoading(false);
    } else if (result.url) {
      // Redirect to Stripe onboarding
      window.location.href = result.url;
    }
  };

  const handleCheckStatus = async () => {
    setIsLoading(true);
    const result = await checkConnectStatusAction();
    setIsLoading(false);
    
    if (result.data?.onboarded) {
      window.location.reload();
    }
  };

  const handleOpenDashboard = async () => {
    setIsLoading(true);
    setError(null);

    const result = await createDashboardLoginLinkAction();

    if (result.error) {
      setError(result.error);
      setIsLoading(false);
    } else if (result.url) {
      window.open(result.url, '_blank');
      setIsLoading(false);
    }
  };

  // Not started
  if (!mediator.stripeAccountId) {
    return (
      <div className="card mb-6">
        <div className="flex items-start">
          <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mr-4">
            <svg className="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
            </svg>
          </div>
          <div className="flex-1">
            <h3 className="text-lg font-semibold mb-2">Set Up Payment Account</h3>
            <p className="text-gray-600 mb-4">
              Connect your bank account to receive payments from clients. We use Stripe to process payments securely.
            </p>
            {error && (
              <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
                {error}
              </div>
            )}
            <button
              onClick={handleStartOnboarding}
              disabled={isLoading}
              className="btn-primary"
            >
              {isLoading ? 'Loading...' : 'Connect Bank Account'}
            </button>
          </div>
        </div>
      </div>
    );
  }

  // Onboarding incomplete
  if (!mediator.stripeOnboarded) {
    return (
      <div className="card mb-6">
        <div className="flex items-start">
          <div className="w-12 h-12 bg-yellow-100 rounded-lg flex items-center justify-center mr-4">
            <svg className="w-6 h-6 text-yellow-600" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
            </svg>
          </div>
          <div className="flex-1">
            <h3 className="text-lg font-semibold mb-2">Complete Payment Setup</h3>
            <p className="text-gray-600 mb-4">
              Your payment account setup is incomplete. Please complete the onboarding process to receive payments.
            </p>
            {connectStatus && (
              <div className="mb-4 text-sm text-gray-600">
                <p>Status: {connectStatus.detailsSubmitted ? '✓ Details submitted' : '✗ Details needed'}</p>
                <p>Charges: {connectStatus.chargesEnabled ? '✓ Enabled' : '✗ Disabled'}</p>
              </div>
            )}
            {error && (
              <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-800">
                {error}
              </div>
            )}
            <div className="flex items-center space-x-3">
              <button
                onClick={handleStartOnboarding}
                disabled={isLoading}
                className="btn-primary"
              >
                {isLoading ? 'Loading...' : 'Continue Setup'}
              </button>
              <button
                onClick={handleCheckStatus}
                disabled={isLoading}
                className="btn-secondary"
              >
                Check Status
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Onboarded - show summary
  return (
    <div className="card mb-6">
      <div className="flex items-start justify-between">
        <div className="flex items-start">
          <div className="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center mr-4">
            <svg className="w-6 h-6 text-green-600" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
            </svg>
          </div>
          <div>
            <h3 className="text-lg font-semibold mb-2">Payment Account Connected</h3>
            <p className="text-gray-600 mb-2">
              Your bank account is connected and ready to receive payments.
            </p>
            {connectStatus && (
              <div className="text-sm text-gray-600 space-y-1">
                <p>✓ Charges enabled</p>
                <p>✓ Payouts enabled: {connectStatus.payoutsEnabled ? 'Yes' : 'Pending'}</p>
              </div>
            )}
          </div>
        </div>
        <button
          onClick={handleOpenDashboard}
          disabled={isLoading}
          className="btn-secondary whitespace-nowrap"
        >
          {isLoading ? 'Loading...' : 'Stripe Dashboard'}
        </button>
      </div>
    </div>
  );
}
