'use client';

import { useState } from 'react';
import { getSuggestedMediatorsAction } from '@/modules/mediator/search/actions';
import Link from 'next/link';

export function PriceEstimator() {
  const [isOpen, setIsOpen] = useState(false);
  const [estimate, setEstimate] = useState({
    estimatedHours: 4,
    numberOfParties: 2,
    requiresTravel: false,
    travelDistance: 0,
    requiresAccommodation: false,
  });

  const [specialization, setSpecialization] = useState('');
  const [maxBudget, setMaxBudget] = useState('');
  const [suggestions, setSuggestions] = useState<any[]>([]);
  const [isLoading, setIsLoading] = useState(false);

  const getSuggestions = async () => {
    setIsLoading(true);

    const result = await getSuggestedMediatorsAction(
      {
        specialization: specialization || undefined,
        maxBudget: maxBudget ? parseFloat(maxBudget) : undefined,
      },
      estimate
    );

    if (result.success && result.data) {
      setSuggestions(result.data);
    }

    setIsLoading(false);
  };

  if (!isOpen) {
    return (
      <div className="bg-primary-50 rounded-lg p-4 border border-primary-200">
        <h3 className="font-semibold text-primary-900 mb-2">
          💡 Price Estimator
        </h3>
        <p className="text-sm text-primary-800 mb-3">
          Get personalized mediator recommendations based on your needs and budget
        </p>
        <button
          onClick={() => setIsOpen(true)}
          className="text-sm text-primary-600 hover:text-primary-700 font-medium"
        >
          Estimate My Cost →
        </button>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow p-6 border-2 border-primary-200">
      <div className="flex items-center justify-between mb-4">
        <h3 className="font-semibold text-gray-900">Price Estimator</h3>
        <button
          onClick={() => setIsOpen(false)}
          className="text-gray-400 hover:text-gray-600"
        >
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
          </svg>
        </button>
      </div>

      <div className="space-y-4">
        {/* Estimated Hours */}
        <div>
          <label className="label">Estimated Hours</label>
          <input
            type="number"
            value={estimate.estimatedHours}
            onChange={(e) => setEstimate({ ...estimate, estimatedHours: parseInt(e.target.value) || 0 })}
            className="input"
            min="1"
            max="100"
          />
        </div>

        {/* Number of Parties */}
        <div>
          <label className="label">Number of Parties</label>
          <select
            value={estimate.numberOfParties}
            onChange={(e) => setEstimate({ ...estimate, numberOfParties: parseInt(e.target.value) })}
            className="input"
          >
            <option value="2">2 parties</option>
            <option value="3">3 parties</option>
            <option value="4">4 parties</option>
            <option value="5">5+ parties</option>
          </select>
        </div>

        {/* Travel Required */}
        <div>
          <label className="flex items-center">
            <input
              type="checkbox"
              checked={estimate.requiresTravel}
              onChange={(e) => setEstimate({ ...estimate, requiresTravel: e.target.checked })}
              className="mr-2"
            />
            <span className="text-sm font-medium text-gray-700">Requires travel</span>
          </label>
          {estimate.requiresTravel && (
            <input
              type="number"
              value={estimate.travelDistance}
              onChange={(e) => setEstimate({ ...estimate, travelDistance: parseInt(e.target.value) || 0 })}
              className="input mt-2"
              placeholder="Distance (km)"
              min="0"
            />
          )}
        </div>

        {/* Accommodation */}
        <div>
          <label className="flex items-center">
            <input
              type="checkbox"
              checked={estimate.requiresAccommodation}
              onChange={(e) => setEstimate({ ...estimate, requiresAccommodation: e.target.checked })}
              className="mr-2"
            />
            <span className="text-sm font-medium text-gray-700">Requires accommodation</span>
          </label>
        </div>

        {/* Specialization */}
        <div>
          <label className="label">Specialization (optional)</label>
          <input
            type="text"
            value={specialization}
            onChange={(e) => setSpecialization(e.target.value)}
            className="input"
            placeholder="e.g., Family Mediation"
          />
        </div>

        {/* Max Budget */}
        <div>
          <label className="label">Maximum Budget (optional)</label>
          <div className="relative">
            <span className="absolute left-3 top-2 text-gray-500">$</span>
            <input
              type="number"
              value={maxBudget}
              onChange={(e) => setMaxBudget(e.target.value)}
              className="input pl-8"
              placeholder="Your budget"
              min="0"
            />
          </div>
        </div>

        {/* Get Suggestions Button */}
        <button
          onClick={getSuggestions}
          disabled={isLoading}
          className="w-full btn-primary"
        >
          {isLoading ? 'Finding...' : 'Get Suggestions'}
        </button>
      </div>

      {/* Suggestions */}
      {suggestions.length > 0 && (
        <div className="mt-6 pt-6 border-t">
          <h4 className="font-semibold text-gray-900 mb-3">
            Recommended Mediators
          </h4>
          <div className="space-y-3">
            {suggestions.slice(0, 3).map((item) => (
              <Link
                key={item.mediator.id}
                href={`/practices/${item.mediator.slug}`}
                className="block p-3 border rounded-lg hover:border-primary-300 hover:bg-primary-50 transition-colors"
              >
                <div className="flex items-start justify-between">
                  <div className="flex-1">
                    <div className="font-medium text-gray-900">
                      {item.mediator.displayName || `${item.mediator.firstName} ${item.mediator.lastName}`}
                    </div>
                    <div className="text-sm text-gray-600">
                      {item.mediator.yearsExperience} years experience
                    </div>
                    <div className="text-sm font-semibold text-green-600 mt-1">
                      Est. ${item.estimate.totalIncGst.toFixed(2)}
                      {item.estimate.isGstRegistered && ' (inc GST)'}
                    </div>
                  </div>
                  <div className="ml-3">
                    <div className="px-2 py-1 bg-primary-100 text-primary-800 rounded text-xs font-medium">
                      Match: {Math.round(item.score)}%
                    </div>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </div>
      )}
    </div>
  );
}
