'use client';

import { useRouter, useSearchParams } from 'next/navigation';
import { useState } from 'react';

interface SearchFiltersProps {
  filters: {
    specialization?: string;
    minPrice?: number;
    maxPrice?: number;
    maxBudget?: number;
    minExperience?: number;
    search?: string;
  };
  priceRanges: {
    minHourlyRate: number;
    maxHourlyRate: number;
    avgHourlyRate: number;
  };
  specializations: string[];
  languages: string[];
}

export function SearchFilters({
  filters,
  priceRanges,
  specializations,
  languages,
}: SearchFiltersProps) {
  const router = useRouter();
  const searchParams = useSearchParams();

  const [localFilters, setLocalFilters] = useState({
    specialization: filters.specialization || '',
    minPrice: filters.minPrice?.toString() || '',
    maxPrice: filters.maxPrice?.toString() || '',
    maxBudget: filters.maxBudget?.toString() || '',
    minExperience: filters.minExperience?.toString() || '',
    search: filters.search || '',
  });

  const handleFilterChange = (key: string, value: string) => {
    setLocalFilters(prev => ({ ...prev, [key]: value }));
  };

  const applyFilters = () => {
    const params = new URLSearchParams();

    Object.entries(localFilters).forEach(([key, value]) => {
      if (value && value.trim() !== '') {
        params.set(key, value);
      }
    });

    router.push(`/find-mediator?${params.toString()}`);
  };

  const clearFilters = () => {
    setLocalFilters({
      specialization: '',
      minPrice: '',
      maxPrice: '',
      maxBudget: '',
      minExperience: '',
      search: '',
    });
    router.push('/find-mediator');
  };

  return (
    <div className="bg-white rounded-lg shadow p-6 sticky top-4">
      <h2 className="text-lg font-semibold mb-4">Filters</h2>

      <div className="space-y-6">
        {/* Search */}
        <div>
          <label htmlFor="search" className="label">
            Search
          </label>
          <input
            type="text"
            id="search"
            value={localFilters.search}
            onChange={(e) => handleFilterChange('search', e.target.value)}
            className="input"
            placeholder="Name or keyword..."
          />
        </div>

        {/* Specialization */}
        <div>
          <label htmlFor="specialization" className="label">
            Specialization
          </label>
          <select
            id="specialization"
            value={localFilters.specialization}
            onChange={(e) => handleFilterChange('specialization', e.target.value)}
            className="input"
          >
            <option value="">All specializations</option>
            {specializations.map((spec) => (
              <option key={spec} value={spec}>
                {spec}
              </option>
            ))}
          </select>
        </div>

        {/* Price Range */}
        <div>
          <label className="label">Hourly Rate</label>
          <div className="grid grid-cols-2 gap-2">
            <div>
              <input
                type="number"
                value={localFilters.minPrice}
                onChange={(e) => handleFilterChange('minPrice', e.target.value)}
                className="input"
                placeholder="Min"
                min="0"
              />
            </div>
            <div>
              <input
                type="number"
                value={localFilters.maxPrice}
                onChange={(e) => handleFilterChange('maxPrice', e.target.value)}
                className="input"
                placeholder="Max"
                min="0"
              />
            </div>
          </div>
          {priceRanges.minHourlyRate > 0 && (
            <p className="text-xs text-gray-500 mt-1">
              Range: ${priceRanges.minHourlyRate} - ${priceRanges.maxHourlyRate}
            </p>
          )}
        </div>

        {/* Maximum Budget */}
        <div>
          <label htmlFor="maxBudget" className="label">
            Maximum Budget
          </label>
          <div className="relative">
            <span className="absolute left-3 top-2 text-gray-500">$</span>
            <input
              type="number"
              id="maxBudget"
              value={localFilters.maxBudget}
              onChange={(e) => handleFilterChange('maxBudget', e.target.value)}
              className="input pl-8"
              placeholder="Total budget"
              min="0"
            />
          </div>
          <p className="text-xs text-gray-500 mt-1">
            Shows mediators within your total budget
          </p>
        </div>

        {/* Experience */}
        <div>
          <label htmlFor="minExperience" className="label">
            Minimum Experience
          </label>
          <select
            id="minExperience"
            value={localFilters.minExperience}
            onChange={(e) => handleFilterChange('minExperience', e.target.value)}
            className="input"
          >
            <option value="">Any experience</option>
            <option value="5">5+ years</option>
            <option value="10">10+ years</option>
            <option value="15">15+ years</option>
            <option value="20">20+ years</option>
          </select>
        </div>

        {/* Action Buttons */}
        <div className="pt-4 border-t space-y-2">
          <button
            onClick={applyFilters}
            className="w-full btn-primary"
          >
            Apply Filters
          </button>
          <button
            onClick={clearFilters}
            className="w-full btn-secondary"
          >
            Clear All
          </button>
        </div>
      </div>
    </div>
  );
}
