'use client';

import { Mediator, MediatorPricing } from '@prisma/client';
import Link from 'next/link';

type MediatorWithPricing = Mediator & {
  pricing: MediatorPricing | null;
};

interface SearchResultsProps {
  mediators: MediatorWithPricing[];
}

export function SearchResults({ mediators }: SearchResultsProps) {
  if (mediators.length === 0) {
    return (
      <div className="bg-white rounded-lg shadow p-12 text-center">
        <svg
          className="w-16 h-16 text-gray-400 mx-auto mb-4"
          fill="none"
          stroke="currentColor"
          viewBox="0 0 24 24"
        >
          <path
            strokeLinecap="round"
            strokeLinejoin="round"
            strokeWidth={2}
            d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
          />
        </svg>
        <h3 className="text-lg font-medium text-gray-900 mb-2">
          No mediators found
        </h3>
        <p className="text-gray-600">
          Try adjusting your filters to see more results
        </p>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {mediators.map((mediator) => (
        <Link
          key={mediator.id}
          href={`/practices/${mediator.slug}`}
          className="block bg-white rounded-lg shadow hover:shadow-lg transition-shadow p-6"
        >
          <div className="flex items-start gap-6">
            {/* Photo */}
            {mediator.photoUrl ? (
              <img
                src={mediator.photoUrl}
                alt={mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
                className="w-24 h-24 rounded-full object-cover flex-shrink-0"
              />
            ) : (
              <div className="w-24 h-24 rounded-full bg-gray-200 flex items-center justify-center flex-shrink-0">
                <svg className="w-12 h-12 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 9a3 3 0 100-6 3 3 0 000 6zm-7 9a7 7 0 1114 0H3z" clipRule="evenodd" />
                </svg>
              </div>
            )}

            {/* Details */}
            <div className="flex-1 min-w-0">
              <div className="flex items-start justify-between gap-4">
                <div className="flex-1">
                  <h3 className="text-xl font-semibold text-gray-900 mb-1">
                    {mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
                  </h3>

                  <div className="flex items-center gap-3 text-sm text-gray-600 mb-3">
                    {mediator.yearsExperience && (
                      <span>{mediator.yearsExperience} years experience</span>
                    )}
                    {mediator.languages.length > 0 && (
                      <>
                        <span>•</span>
                        <span>{mediator.languages.join(', ')}</span>
                      </>
                    )}
                  </div>

                  {/* Specializations */}
                  {mediator.specializations.length > 0 && (
                    <div className="flex flex-wrap gap-2 mb-3">
                      {mediator.specializations.slice(0, 5).map((spec) => (
                        <span
                          key={spec}
                          className="px-3 py-1 bg-primary-100 text-primary-800 rounded-full text-xs font-medium"
                        >
                          {spec}
                        </span>
                      ))}
                      {mediator.specializations.length > 5 && (
                        <span className="px-3 py-1 bg-gray-100 text-gray-600 rounded-full text-xs">
                          +{mediator.specializations.length - 5} more
                        </span>
                      )}
                    </div>
                  )}

                  {/* Bio preview */}
                  {mediator.bio && (
                    <p className="text-gray-700 line-clamp-2 text-sm">
                      {mediator.bio}
                    </p>
                  )}
                </div>

                {/* Pricing */}
                {mediator.pricing && (
                  <div className="flex-shrink-0 text-right">
                    <div className="text-2xl font-bold text-gray-900">
                      ${mediator.pricing.hourlyRate.toString()}
                      <span className="text-sm font-normal text-gray-600">/hr</span>
                    </div>
                    <div className="text-xs text-gray-500 mt-1">
                      {mediator.pricing.hourlyRateBasis === 'PER_PARTY' ? 'per party' : 'total'}
                    </div>
                    {mediator.isGstRegistered && (
                      <div className="text-xs text-gray-500 mt-1">
                        (inc GST)
                      </div>
                    )}
                    {mediator.pricing.preMediationFee && (
                      <div className="text-xs text-gray-600 mt-2">
                        Pre-mediation: ${mediator.pricing.preMediationFee.toString()}
                      </div>
                    )}
                  </div>
                )}
              </div>
            </div>
          </div>
        </Link>
      ))}
    </div>
  );
}
