import { prisma } from './db';

export interface EmailProvider {
  send(params: {
    to: string;
    subject: string;
    html: string;
  }): Promise<void>;
}

// Console logger for development
export class ConsoleEmailProvider implements EmailProvider {
  async send(params: { to: string; subject: string; html: string }) {
    console.log('\n=== EMAIL ===');
    console.log('To:', params.to);
    console.log('Subject:', params.subject);
    console.log('Body:\n', params.html);
    console.log('=============\n');
  }
}

// SendGrid for production (stub - implement later)
export class SendGridEmailProvider implements EmailProvider {
  private apiKey: string;

  constructor(apiKey: string) {
    this.apiKey = apiKey;
  }

  async send(params: { to: string; subject: string; html: string }) {
    // TODO: Implement with @sendgrid/mail
    // const sgMail = require('@sendgrid/mail');
    // sgMail.setApiKey(this.apiKey);
    // await sgMail.send({
    //   to: params.to,
    //   from: 'noreply@medi8.com.au',
    //   subject: params.subject,
    //   html: params.html,
    // });
    throw new Error('SendGrid not implemented yet. Set EMAIL_PROVIDER=console');
  }
}

export function getEmailProvider(): EmailProvider {
  const provider = process.env.EMAIL_PROVIDER || 'console';
  
  switch (provider) {
    case 'sendgrid':
      if (!process.env.SENDGRID_API_KEY) {
        throw new Error('SENDGRID_API_KEY environment variable is required');
      }
      return new SendGridEmailProvider(process.env.SENDGRID_API_KEY);
    default:
      return new ConsoleEmailProvider();
  }
}

interface SendEmailParams {
  to: string;
  template: string;
  data: Record<string, any>;
}

export async function sendEmail(params: SendEmailParams) {
  const template = await prisma.notificationTemplate.findUnique({
    where: { key: params.template },
  });

  if (!template) {
    throw new Error(`Email template '${params.template}' not found`);
  }

  if (!template.isActive) {
    console.warn(`Email template '${params.template}' is inactive, skipping send`);
    return;
  }

  // Simple template rendering - replace {{placeholders}}
  let subject = template.subject;
  let html = template.bodyTemplate;

  for (const [key, value] of Object.entries(params.data)) {
    const regex = new RegExp(`{{${key}}}`, 'g');
    subject = subject.replace(regex, String(value));
    html = html.replace(regex, String(value));
  }

  // Add app URL if not provided
  if (!params.data.appUrl) {
    const appUrl = process.env.APP_URL || 'http://localhost:3000';
    html = html.replace(/{{appUrl}}/g, appUrl);
  }

  const provider = getEmailProvider();
  
  try {
    await provider.send({
      to: params.to,
      subject,
      html: html.replace(/\n/g, '<br>'), // Convert newlines to <br>
    });

    // Log notification
    await prisma.notificationLog.create({
      data: {
        recipientEmail: params.to,
        templateKey: params.template,
        subject,
        body: html,
        sentAt: new Date(),
      },
    });
  } catch (error) {
    // Log failure
    await prisma.notificationLog.create({
      data: {
        recipientEmail: params.to,
        templateKey: params.template,
        subject,
        body: html,
        failedAt: new Date(),
        errorMessage: error instanceof Error ? error.message : 'Unknown error',
      },
    });
    
    throw error;
  }
}
