import axios, { AxiosInstance } from 'axios';

export interface HubSpotConfig {
  apiKey: string;
  enabled: boolean;
}

export interface HubSpotContact {
  email: string;
  firstname?: string;
  lastname?: string;
  phone?: string;
  lifecyclestage?: string;
}

export interface HubSpotDeal {
  dealname: string;
  amount?: number;
  dealstage: string;
  pipeline?: string;
  closedate?: string;
}

export interface HubSpotDealStage {
  NEW_ENQUIRY: string;
  DEPOSIT_PAID: string;
  APPROVED_TO_PROCEED: string;
  BALANCE_PAID: string;
  COMPLETED: string;
  CANCELLED: string;
}

// HubSpot deal stages - customize these IDs based on your HubSpot setup
export const HUBSPOT_DEAL_STAGES: HubSpotDealStage = {
  NEW_ENQUIRY: 'appointmentscheduled', // Stage 1
  DEPOSIT_PAID: 'qualifiedtobuy', // Stage 2
  APPROVED_TO_PROCEED: 'presentationscheduled', // Stage 3
  BALANCE_PAID: 'decisionmakerboughtin', // Stage 4
  COMPLETED: 'closedwon', // Won
  CANCELLED: 'closedlost', // Lost
};

export class HubSpotClient {
  private client: AxiosInstance;
  private enabled: boolean;

  constructor(config: HubSpotConfig) {
    this.enabled = config.enabled && !!config.apiKey;

    this.client = axios.create({
      baseURL: 'https://api.hubapi.com',
      headers: {
        'Authorization': `Bearer ${config.apiKey}`,
        'Content-Type': 'application/json',
      },
      timeout: 10000, // 10 second timeout
    });
  }

  /**
   * Check if HubSpot integration is enabled
   */
  isEnabled(): boolean {
    return this.enabled;
  }

  /**
   * Create or update a contact
   */
  async upsertContact(contact: HubSpotContact): Promise<string | null> {
    if (!this.enabled) {
      console.log('[HubSpot] Skipped - integration disabled');
      return null;
    }

    try {
      // Try to find existing contact by email
      const searchResponse = await this.client.post(
        '/crm/v3/objects/contacts/search',
        {
          filterGroups: [
            {
              filters: [
                {
                  propertyName: 'email',
                  operator: 'EQ',
                  value: contact.email,
                },
              ],
            },
          ],
        }
      );

      if (searchResponse.data.results && searchResponse.data.results.length > 0) {
        // Update existing contact
        const contactId = searchResponse.data.results[0].id;
        await this.client.patch(`/crm/v3/objects/contacts/${contactId}`, {
          properties: contact,
        });
        console.log(`[HubSpot] Updated contact: ${contactId}`);
        return contactId;
      } else {
        // Create new contact
        const createResponse = await this.client.post(
          '/crm/v3/objects/contacts',
          {
            properties: contact,
          }
        );
        console.log(`[HubSpot] Created contact: ${createResponse.data.id}`);
        return createResponse.data.id;
      }
    } catch (error) {
      console.error('[HubSpot] Failed to upsert contact:', error);
      return null;
    }
  }

  /**
   * Create a deal
   */
  async createDeal(deal: HubSpotDeal, contactId?: string): Promise<string | null> {
    if (!this.enabled) {
      console.log('[HubSpot] Skipped - integration disabled');
      return null;
    }

    try {
      const createResponse = await this.client.post('/crm/v3/objects/deals', {
        properties: deal,
      });

      const dealId = createResponse.data.id;
      console.log(`[HubSpot] Created deal: ${dealId}`);

      // Associate deal with contact if provided
      if (contactId) {
        await this.associateDealWithContact(dealId, contactId);
      }

      return dealId;
    } catch (error) {
      console.error('[HubSpot] Failed to create deal:', error);
      return null;
    }
  }

  /**
   * Update deal stage
   */
  async updateDealStage(dealId: string, stage: string): Promise<boolean> {
    if (!this.enabled) {
      console.log('[HubSpot] Skipped - integration disabled');
      return false;
    }

    try {
      await this.client.patch(`/crm/v3/objects/deals/${dealId}`, {
        properties: {
          dealstage: stage,
        },
      });
      console.log(`[HubSpot] Updated deal ${dealId} to stage: ${stage}`);
      return true;
    } catch (error) {
      console.error('[HubSpot] Failed to update deal stage:', error);
      return false;
    }
  }

  /**
   * Update deal properties
   */
  async updateDeal(dealId: string, properties: Partial<HubSpotDeal>): Promise<boolean> {
    if (!this.enabled) {
      console.log('[HubSpot] Skipped - integration disabled');
      return false;
    }

    try {
      await this.client.patch(`/crm/v3/objects/deals/${dealId}`, {
        properties,
      });
      console.log(`[HubSpot] Updated deal: ${dealId}`);
      return true;
    } catch (error) {
      console.error('[HubSpot] Failed to update deal:', error);
      return false;
    }
  }

  /**
   * Associate a deal with a contact
   */
  async associateDealWithContact(dealId: string, contactId: string): Promise<boolean> {
    if (!this.enabled) {
      console.log('[HubSpot] Skipped - integration disabled');
      return false;
    }

    try {
      await this.client.put(
        `/crm/v3/objects/deals/${dealId}/associations/contacts/${contactId}/deal_to_contact`
      );
      console.log(`[HubSpot] Associated deal ${dealId} with contact ${contactId}`);
      return true;
    } catch (error) {
      console.error('[HubSpot] Failed to associate deal with contact:', error);
      return false;
    }
  }

  /**
   * Add a note to a deal
   */
  async addNoteToContact(contactId: string, noteBody: string): Promise<boolean> {
    if (!this.enabled) {
      console.log('[HubSpot] Skipped - integration disabled');
      return false;
    }

    try {
      const noteResponse = await this.client.post('/crm/v3/objects/notes', {
        properties: {
          hs_note_body: noteBody,
          hs_timestamp: new Date().getTime(),
        },
      });

      const noteId = noteResponse.data.id;

      // Associate note with contact
      await this.client.put(
        `/crm/v3/objects/notes/${noteId}/associations/contacts/${contactId}/note_to_contact`
      );

      console.log(`[HubSpot] Added note to contact ${contactId}`);
      return true;
    } catch (error) {
      console.error('[HubSpot] Failed to add note:', error);
      return false;
    }
  }

  /**
   * Get deal by ID
   */
  async getDeal(dealId: string): Promise<any> {
    if (!this.enabled) {
      return null;
    }

    try {
      const response = await this.client.get(`/crm/v3/objects/deals/${dealId}`);
      return response.data;
    } catch (error) {
      console.error('[HubSpot] Failed to get deal:', error);
      return null;
    }
  }

  /**
   * Test connection to HubSpot
   */
  async testConnection(): Promise<boolean> {
    if (!this.enabled) {
      console.log('[HubSpot] Integration is disabled');
      return false;
    }

    try {
      await this.client.get('/crm/v3/objects/contacts', {
        params: { limit: 1 },
      });
      console.log('[HubSpot] Connection test successful');
      return true;
    } catch (error) {
      console.error('[HubSpot] Connection test failed:', error);
      return false;
    }
  }
}

/**
 * Get HubSpot client instance
 */
export function getHubSpotClient(): HubSpotClient {
  const config: HubSpotConfig = {
    apiKey: process.env.HUBSPOT_API_KEY || '',
    enabled: process.env.HUBSPOT_ENABLED === 'true',
  };

  return new HubSpotClient(config);
}
