import { writeFile, readFile, unlink, mkdir } from 'fs/promises';
import { join } from 'path';
import { existsSync } from 'fs';

export interface StorageProvider {
  upload(file: File, path: string): Promise<string>;
  uploadBuffer(buffer: Buffer, path: string, mimeType: string): Promise<string>;
  download(path: string): Promise<Buffer>;
  delete(path: string): Promise<void>;
  getUrl(path: string): string;
  exists(path: string): Promise<boolean>;
}

// Local filesystem storage for development
export class LocalStorageProvider implements StorageProvider {
  private baseDir = join(process.cwd(), 'public', 'uploads');

  async upload(file: File, path: string): Promise<string> {
    const fullPath = join(this.baseDir, path);
    
    // Ensure directory exists
    const dir = fullPath.substring(0, fullPath.lastIndexOf('/'));
    if (!existsSync(dir)) {
      await mkdir(dir, { recursive: true });
    }
    
    const buffer = Buffer.from(await file.arrayBuffer());
    await writeFile(fullPath, buffer);
    
    return path;
  }

  async uploadBuffer(buffer: Buffer, path: string, mimeType: string): Promise<string> {
    const fullPath = join(this.baseDir, path);
    
    // Ensure directory exists
    const dir = fullPath.substring(0, fullPath.lastIndexOf('/'));
    if (!existsSync(dir)) {
      await mkdir(dir, { recursive: true });
    }
    
    await writeFile(fullPath, buffer);
    
    return path;
  }

  async download(path: string): Promise<Buffer> {
    const fullPath = join(this.baseDir, path);
    return readFile(fullPath);
  }

  async delete(path: string): Promise<void> {
    const fullPath = join(this.baseDir, path);
    await unlink(fullPath);
  }

  getUrl(path: string): string {
    return `/uploads/${path}`;
  }

  async exists(path: string): Promise<boolean> {
    const fullPath = join(this.baseDir, path);
    return existsSync(fullPath);
  }
}

// S3 storage for production (stub - implement later)
export class S3StorageProvider implements StorageProvider {
  private bucket: string;
  private region: string;

  constructor(bucket: string, region: string) {
    this.bucket = bucket;
    this.region = region;
  }

  async upload(file: File, path: string): Promise<string> {
    // TODO: Implement with AWS SDK
    // const s3 = new S3Client({ region: this.region });
    // const buffer = Buffer.from(await file.arrayBuffer());
    // await s3.send(new PutObjectCommand({
    //   Bucket: this.bucket,
    //   Key: path,
    //   Body: buffer,
    //   ContentType: file.type,
    // }));
    throw new Error('S3 not implemented yet. Set STORAGE_PROVIDER=local');
  }

  async uploadBuffer(buffer: Buffer, path: string, mimeType: string): Promise<string> {
    // TODO: Implement with AWS SDK
    throw new Error('S3 not implemented yet. Set STORAGE_PROVIDER=local');
  }

  async download(path: string): Promise<Buffer> {
    // TODO: Implement with AWS SDK
    throw new Error('S3 not implemented yet. Set STORAGE_PROVIDER=local');
  }

  async delete(path: string): Promise<void> {
    // TODO: Implement with AWS SDK
    throw new Error('S3 not implemented yet. Set STORAGE_PROVIDER=local');
  }

  getUrl(path: string): string {
    return `https://${this.bucket}.s3.${this.region}.amazonaws.com/${path}`;
  }

  async exists(path: string): Promise<boolean> {
    // TODO: Implement with AWS SDK
    throw new Error('S3 not implemented yet. Set STORAGE_PROVIDER=local');
  }
}

export function getStorageProvider(): StorageProvider {
  const provider = process.env.STORAGE_PROVIDER || 'local';
  
  switch (provider) {
    case 's3':
      if (!process.env.AWS_S3_BUCKET || !process.env.AWS_REGION) {
        throw new Error('AWS_S3_BUCKET and AWS_REGION environment variables are required');
      }
      return new S3StorageProvider(
        process.env.AWS_S3_BUCKET,
        process.env.AWS_REGION
      );
    default:
      return new LocalStorageProvider();
  }
}

/**
 * Generate a unique file path for storage
 */
export function generateFilePath(
  folder: string,
  originalFilename: string,
  prefix?: string
): string {
  const timestamp = Date.now();
  const random = Math.random().toString(36).substring(7);
  const ext = originalFilename.split('.').pop();
  const name = prefix ? `${prefix}-${timestamp}-${random}` : `${timestamp}-${random}`;
  
  return `${folder}/${name}.${ext}`;
}
