import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';
import { prisma } from '@/lib/db';

export async function middleware(request: NextRequest) {
  const token = request.cookies.get('session_token')?.value;
  const pathname = request.nextUrl.pathname;

  // Public routes that don't require authentication
  const publicRoutes = [
    '/',
    '/login',
    '/register/client',
    '/register/mediator',
  ];

  const isPublicRoute = 
    publicRoutes.includes(pathname) ||
    pathname.startsWith('/_next') ||
    pathname.startsWith('/api/webhooks') ||
    pathname.startsWith('/practices/') || // Public practice pages
    pathname.startsWith('/acknowledge/') || // Document acknowledgement
    pathname.startsWith('/uploads/'); // Public file access

  // Allow public routes
  if (isPublicRoute) {
    return NextResponse.next();
  }

  // Protected routes require authentication
  if (!token) {
    return NextResponse.redirect(new URL('/login', request.url));
  }

  // Validate session
  const session = await prisma.session.findUnique({
    where: { token },
    include: {
      user: {
        include: {
          mediator: true,
          client: true,
        },
      },
    },
  });

  // Invalid or expired session
  if (!session || session.expiresAt < new Date()) {
    const response = NextResponse.redirect(new URL('/login', request.url));
    response.cookies.delete('session_token');
    return response;
  }

  const user = session.user;

  // Role-based access control
  if (pathname.startsWith('/admin') && user.role !== 'ADMIN') {
    return NextResponse.redirect(new URL('/login', request.url));
  }

  if (pathname.startsWith('/mediator') && user.role !== 'MEDIATOR') {
    return NextResponse.redirect(new URL('/login', request.url));
  }

  if (pathname.startsWith('/client') && user.role !== 'CLIENT') {
    return NextResponse.redirect(new URL('/login', request.url));
  }

  return NextResponse.next();
}

export const config = {
  matcher: [
    /*
     * Match all request paths except for the ones starting with:
     * - _next/static (static files)
     * - _next/image (image optimization files)
     * - favicon.ico (favicon file)
     */
    '/((?!_next/static|_next/image|favicon.ico).*)',
  ],
};
