'use server';

import { revalidatePath } from 'next/cache';
import { redirect } from 'next/navigation';
import { ApprovalService } from './service';

const approvalService = new ApprovalService();

interface ActionResult {
  success?: boolean;
  error?: string;
}

/**
 * Approve a mediator
 */
export async function approveMediatorAction(
  mediatorId: string,
  adminId: string,
  notes?: string
): Promise<ActionResult> {
  try {
    await approvalService.approveMediator(mediatorId, adminId, notes);
    revalidatePath('/admin/approvals');
    return { success: true };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to approve mediator' };
  }
}

/**
 * Reject a mediator
 */
export async function rejectMediatorAction(
  mediatorId: string,
  adminId: string,
  formData: FormData
): Promise<ActionResult> {
  try {
    const reason = formData.get('reason') as string;
    
    if (!reason || reason.trim().length === 0) {
      return { error: 'Rejection reason is required' };
    }

    await approvalService.rejectMediator(mediatorId, adminId, reason);
    revalidatePath('/admin/approvals');
    return { success: true };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to reject mediator' };
  }
}

/**
 * Publish an approved mediator
 */
export async function publishMediatorAction(
  mediatorId: string,
  adminId: string
): Promise<ActionResult> {
  try {
    await approvalService.publishMediator(mediatorId, adminId);
    revalidatePath('/admin/approvals');
    revalidatePath(`/admin/mediators/${mediatorId}`);
    return { success: true };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to publish mediator' };
  }
}

/**
 * Suspend a mediator
 */
export async function suspendMediatorAction(
  mediatorId: string,
  adminId: string,
  formData: FormData
): Promise<ActionResult> {
  try {
    const reason = formData.get('reason') as string;
    
    if (!reason || reason.trim().length === 0) {
      return { error: 'Suspension reason is required' };
    }

    await approvalService.suspendMediator(mediatorId, adminId, reason);
    revalidatePath('/admin/approvals');
    revalidatePath(`/admin/mediators/${mediatorId}`);
    return { success: true };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to suspend mediator' };
  }
}

/**
 * Unsuspend a mediator
 */
export async function unsuspendMediatorAction(
  mediatorId: string,
  adminId: string
): Promise<ActionResult> {
  try {
    await approvalService.unsuspendMediator(mediatorId, adminId);
    revalidatePath('/admin/approvals');
    revalidatePath(`/admin/mediators/${mediatorId}`);
    return { success: true };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to unsuspend mediator' };
  }
}
