import { prisma } from '@/lib/db';
import { MediatorStatus } from '@prisma/client';
import { NotFoundError, ForbiddenError } from '@/modules/shared/errors';
import { createAuditLog } from '@/modules/shared/audit';
import { sendEmail } from '@/lib/email';

export class ApprovalService {
  /**
   * Get pending mediator approvals
   */
  async getPendingApprovals() {
    return prisma.mediator.findMany({
      where: { status: 'PENDING_APPROVAL' },
      include: {
        user: { 
          select: { 
            email: true, 
            createdAt: true 
          } 
        },
        pricing: true,
      },
      orderBy: { createdAt: 'asc' },
    });
  }

  /**
   * Get mediator details for review
   */
  async getMediatorForReview(mediatorId: string) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: {
        user: {
          select: {
            email: true,
            createdAt: true,
            lastLoginAt: true,
          },
        },
        pricing: true,
      },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    return mediator;
  }

  /**
   * Approve a mediator
   */
  async approveMediator(
    mediatorId: string,
    adminId: string,
    notes?: string
  ) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: { user: true },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    if (mediator.status === 'APPROVED' || mediator.status === 'PUBLISHED') {
      throw new ForbiddenError('Mediator is already approved');
    }

    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'APPROVED',
        approvalNotes: notes,
        approvedBy: adminId,
        approvedAt: new Date(),
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'MEDIATOR_APPROVE',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { notes },
    });

    // Send approval email
    await sendEmail({
      to: mediator.user.email,
      template: 'MEDIATOR_APPROVED',
      data: {
        firstName: mediator.firstName,
        slug: mediator.slug,
        appUrl: process.env.APP_URL || 'http://localhost:3000',
      },
    });

    return updated;
  }

  /**
   * Reject a mediator
   */
  async rejectMediator(
    mediatorId: string,
    adminId: string,
    reason: string
  ) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: { user: true },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'REJECTED',
        approvalNotes: reason,
        approvedBy: adminId,
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'MEDIATOR_REJECT',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { reason },
    });

    // Send rejection email
    await sendEmail({
      to: mediator.user.email,
      template: 'MEDIATOR_REJECTED',
      data: {
        firstName: mediator.firstName,
        reason,
        appUrl: process.env.APP_URL || 'http://localhost:3000',
      },
    });

    return updated;
  }

  /**
   * Publish an approved mediator profile
   */
  async publishMediator(mediatorId: string, adminId: string) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: { user: true },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    if (mediator.status !== 'APPROVED') {
      throw new ForbiddenError('Mediator must be approved before publishing');
    }

    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'PUBLISHED',
        publishedAt: new Date(),
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'PROFILE_PUBLISH',
      resource: 'mediator',
      resourceId: mediatorId,
    });

    // Send published notification
    await sendEmail({
      to: mediator.user.email,
      template: 'MEDIATOR_PUBLISHED',
      data: {
        firstName: mediator.firstName,
        slug: mediator.slug,
        practiceUrl: `${process.env.APP_URL}/practices/${mediator.slug}`,
        appUrl: process.env.APP_URL || 'http://localhost:3000',
      },
    });

    return updated;
  }

  /**
   * Suspend a mediator
   */
  async suspendMediator(
    mediatorId: string,
    adminId: string,
    reason: string
  ) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: { user: true },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'SUSPENDED',
        approvalNotes: reason,
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'MEDIATOR_SUSPEND',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { reason },
    });

    // Send suspension email
    await sendEmail({
      to: mediator.user.email,
      template: 'MEDIATOR_SUSPENDED',
      data: {
        firstName: mediator.firstName,
        reason,
        appUrl: process.env.APP_URL || 'http://localhost:3000',
      },
    });

    return updated;
  }

  /**
   * Unsuspend a mediator (restore to previous status)
   */
  async unsuspendMediator(
    mediatorId: string,
    adminId: string
  ) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    if (mediator.status !== 'SUSPENDED') {
      throw new ForbiddenError('Mediator is not suspended');
    }

    // Restore to approved status (can be published again later)
    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'APPROVED',
        approvalNotes: null,
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'ADMIN_ACTION',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { action: 'unsuspended' },
    });

    return updated;
  }

  /**
   * Get all mediators with filters
   */
  async getAllMediators(filters?: {
    status?: MediatorStatus;
    search?: string;
  }) {
    return prisma.mediator.findMany({
      where: {
        ...(filters?.status && { status: filters.status }),
        ...(filters?.search && {
          OR: [
            { firstName: { contains: filters.search, mode: 'insensitive' } },
            { lastName: { contains: filters.search, mode: 'insensitive' } },
            { user: { email: { contains: filters.search, mode: 'insensitive' } } },
          ],
        }),
      },
      include: {
        user: {
          select: {
            email: true,
            createdAt: true,
          },
        },
        pricing: true,
      },
      orderBy: { createdAt: 'desc' },
    });
  }
}
