'use server';

import { cookies } from 'next/headers';
import { redirect } from 'next/navigation';
import { AuthService } from './service';
import { 
  registerClientSchema, 
  registerMediatorSchema, 
  loginSchema 
} from './validation';
import { createAuditLog } from '@/modules/shared/audit';
import { ZodError } from 'zod';

const authService = new AuthService();

interface ActionResult {
  error?: string;
  success?: boolean;
}

/**
 * Register a new client
 */
export async function registerClientAction(formData: FormData): Promise<ActionResult> {
  try {
    const data = {
      email: formData.get('email') as string,
      password: formData.get('password') as string,
      firstName: formData.get('firstName') as string,
      lastName: formData.get('lastName') as string,
      phone: formData.get('phone') as string || undefined,
    };

    const validated = registerClientSchema.parse(data);
    const user = await authService.registerClient(validated);
    const session = await authService.createSession(user.id);

    // Set session cookie
    cookies().set('session_token', session.token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 7 * 24 * 60 * 60, // 7 days
      path: '/',
    });

    await createAuditLog({
      userId: user.id,
      action: 'USER_REGISTER',
      resource: 'user',
      resourceId: user.id,
      metadata: { role: 'CLIENT' },
    });

    redirect('/client/dashboard');
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Registration failed' };
  }
}

/**
 * Register a new mediator
 */
export async function registerMediatorAction(formData: FormData): Promise<ActionResult> {
  try {
    const data = {
      email: formData.get('email') as string,
      password: formData.get('password') as string,
      firstName: formData.get('firstName') as string,
      lastName: formData.get('lastName') as string,
      phone: formData.get('phone') as string || undefined,
      isGstRegistered: formData.get('isGstRegistered') === 'true',
      gstNumber: formData.get('gstNumber') as string || undefined,
    };

    const validated = registerMediatorSchema.parse(data);
    const user = await authService.registerMediator(validated);
    const session = await authService.createSession(user.id);

    // Set session cookie
    cookies().set('session_token', session.token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 7 * 24 * 60 * 60,
      path: '/',
    });

    await createAuditLog({
      userId: user.id,
      action: 'USER_REGISTER',
      resource: 'mediator',
      resourceId: user.mediator!.id,
      metadata: {
        status: 'PENDING_APPROVAL',
        isGstRegistered: validated.isGstRegistered,
      },
    });

    redirect('/mediator/dashboard');
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Registration failed' };
  }
}

/**
 * Login user
 */
export async function loginAction(formData: FormData): Promise<ActionResult> {
  try {
    const data = {
      email: formData.get('email') as string,
      password: formData.get('password') as string,
    };

    const validated = loginSchema.parse(data);
    const user = await authService.login(validated.email, validated.password);
    const session = await authService.createSession(user.id);

    // Set session cookie
    cookies().set('session_token', session.token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 7 * 24 * 60 * 60,
      path: '/',
    });

    await createAuditLog({
      userId: user.id,
      action: 'USER_LOGIN',
      resource: 'user',
      resourceId: user.id,
    });

    // Redirect based on role
    switch (user.role) {
      case 'ADMIN':
        redirect('/admin/dashboard');
      case 'MEDIATOR':
        redirect('/mediator/dashboard');
      case 'CLIENT':
        redirect('/client/dashboard');
      default:
        redirect('/');
    }
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    return { error: 'Invalid email or password' };
  }
}

/**
 * Logout user
 */
export async function logoutAction() {
  const token = cookies().get('session_token')?.value;
  
  if (token) {
    try {
      await authService.deleteSession(token);
    } catch (error) {
      // Session might already be deleted, ignore error
    }
  }

  cookies().delete('session_token');
  redirect('/login');
}

/**
 * Get current authenticated user
 */
export async function getCurrentUser() {
  const token = cookies().get('session_token')?.value;
  
  if (!token) {
    return null;
  }

  return authService.getSessionUser(token);
}
