import bcrypt from 'bcrypt';
import { prisma } from '@/lib/db';
import { UserRole } from '@prisma/client';
import { UnauthorizedError, ValidationError } from '@/modules/shared/errors';
import type { 
  RegisterClientInput, 
  RegisterMediatorInput,
  UserWithRelations 
} from './types';

export class AuthService {
  /**
   * Register a new client user
   */
  async registerClient(data: RegisterClientInput): Promise<UserWithRelations> {
    // Check if email exists
    const existingUser = await prisma.user.findUnique({
      where: { email: data.email.toLowerCase() },
    });

    if (existingUser) {
      throw new ValidationError('Email already registered');
    }

    // Hash password
    const passwordHash = await bcrypt.hash(data.password, 10);

    // Create user with client profile
    const user = await prisma.user.create({
      data: {
        email: data.email.toLowerCase(),
        passwordHash,
        role: UserRole.CLIENT,
        client: {
          create: {
            firstName: data.firstName,
            lastName: data.lastName,
            phone: data.phone,
          },
        },
      },
      include: { 
        client: true,
        mediator: true,
      },
    });

    return user;
  }

  /**
   * Register a new mediator user
   */
  async registerMediator(data: RegisterMediatorInput): Promise<UserWithRelations> {
    // Check if email exists
    const existingUser = await prisma.user.findUnique({
      where: { email: data.email.toLowerCase() },
    });

    if (existingUser) {
      throw new ValidationError('Email already registered');
    }

    // Hash password
    const passwordHash = await bcrypt.hash(data.password, 10);

    // Generate unique slug
    const slug = await this.generateUniqueSlug(data.firstName, data.lastName);

    // Calculate trial end date
    const trialDays = await this.getTrialPeriodDays();
    const trialEnd = new Date();
    trialEnd.setDate(trialEnd.getDate() + trialDays);

    // Create user with mediator profile
    const user = await prisma.user.create({
      data: {
        email: data.email.toLowerCase(),
        passwordHash,
        role: UserRole.MEDIATOR,
        mediator: {
          create: {
            slug,
            firstName: data.firstName,
            lastName: data.lastName,
            phone: data.phone,
            status: 'PENDING_APPROVAL',
            subscriptionStatus: 'TRIAL',
            subscriptionStartedAt: new Date(),
            subscriptionEndsAt: trialEnd,
            currentPeriodStart: new Date(),
            currentPeriodEnd: trialEnd,
            isGstRegistered: data.isGstRegistered || false,
            gstNumber: data.gstNumber,
            gstRegisteredAt: data.gstNumber ? new Date() : null,
          },
        },
      },
      include: { 
        mediator: true,
        client: true,
      },
    });

    return user;
  }

  /**
   * Authenticate user with email and password
   */
  async login(email: string, password: string): Promise<UserWithRelations> {
    const user = await prisma.user.findUnique({
      where: { email: email.toLowerCase() },
      include: {
        mediator: true,
        client: true,
      },
    });

    if (!user) {
      throw new UnauthorizedError('Invalid credentials');
    }

    if (!user.isActive) {
      throw new UnauthorizedError('Account is inactive');
    }

    // Verify password
    const validPassword = await bcrypt.compare(password, user.passwordHash);
    if (!validPassword) {
      throw new UnauthorizedError('Invalid credentials');
    }

    // Update last login
    await prisma.user.update({
      where: { id: user.id },
      data: { lastLoginAt: new Date() },
    });

    return user;
  }

  /**
   * Create a new session for a user
   */
  async createSession(userId: string) {
    const token = crypto.randomUUID();
    const expiresAt = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000); // 7 days

    const session = await prisma.session.create({
      data: {
        userId,
        token,
        expiresAt,
      },
    });

    return session;
  }

  /**
   * Get user from session token
   */
  async getSessionUser(token: string): Promise<UserWithRelations | null> {
    const session = await prisma.session.findUnique({
      where: { token },
      include: {
        user: {
          include: {
            mediator: true,
            client: true,
          },
        },
      },
    });

    if (!session || session.expiresAt < new Date()) {
      return null;
    }

    return session.user;
  }

  /**
   * Delete a session (logout)
   */
  async deleteSession(token: string) {
    await prisma.session.delete({
      where: { token },
    });
  }

  /**
   * Clean up expired sessions (can be run as a cron job)
   */
  async cleanupExpiredSessions() {
    const result = await prisma.session.deleteMany({
      where: {
        expiresAt: {
          lt: new Date(),
        },
      },
    });

    return result.count;
  }

  /**
   * Generate a unique slug for a mediator
   */
  private async generateUniqueSlug(firstName: string, lastName: string): Promise<string> {
    const baseSlug = `${firstName}-${lastName}`
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, '-')
      .replace(/^-+|-+$/g, '');
    
    let slug = baseSlug;
    let counter = 1;
    
    while (await prisma.mediator.findUnique({ where: { slug } })) {
      slug = `${baseSlug}-${counter}`;
      counter++;
    }

    return slug;
  }

  /**
   * Get trial period from platform config
   */
  private async getTrialPeriodDays(): Promise<number> {
    const config = await prisma.platformConfig.findUnique({
      where: { key: 'TRIAL_PERIOD_DAYS' },
    });
    return parseInt(config?.value || '30');
  }
}
