'use server';

import { revalidatePath } from 'next/cache';
import { redirect } from 'next/navigation';
import { CaseService } from './service';
import { 
  createInquirySchema, 
  updateCaseSchema, 
  addNoteSchema,
  addPartySchema 
} from './validation';
import { getCurrentUser } from '@/modules/auth/actions';
import { CaseStatus } from '@prisma/client';
import { ZodError } from 'zod';

const caseService = new CaseService();

interface ActionResult {
  error?: string;
  success?: boolean;
  case?: any;
  caseId?: string;
}

/**
 * Create inquiry from public form (client must be logged in)
 */
export async function createInquiryAction(
  mediatorId: string,
  formData: FormData
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'CLIENT' || !user.client) {
      return { error: 'You must be logged in as a client to submit an inquiry' };
    }

    const data = {
      mediatorId,
      clientId: user.client.id,
      title: formData.get('title') as string,
      description: formData.get('description') as string,
      disputeType: formData.get('disputeType') as string || undefined,
      estimatedHours: formData.get('estimatedHours') 
        ? parseFloat(formData.get('estimatedHours') as string)
        : undefined,
      requiresTravel: formData.get('requiresTravel') === 'true',
      travelDistance: formData.get('travelDistance')
        ? parseInt(formData.get('travelDistance') as string)
        : undefined,
      requiresAccommodation: formData.get('requiresAccommodation') === 'true',
    };

    const validated = createInquirySchema.parse(data);

    const caseRecord = await caseService.createInquiry(validated);

    revalidatePath('/client/cases');
    return { success: true, caseId: caseRecord.id };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to submit inquiry' };
  }
}

/**
 * Update case details (mediator only)
 */
export async function updateCaseAction(
  caseId: string,
  formData: FormData
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR') {
      return { error: 'Unauthorized' };
    }

    const data = {
      title: formData.get('title') as string || undefined,
      description: formData.get('description') as string || undefined,
      disputeType: formData.get('disputeType') as string || undefined,
      estimatedHours: formData.get('estimatedHours')
        ? parseFloat(formData.get('estimatedHours') as string)
        : undefined,
      scheduledDate: formData.get('scheduledDate')
        ? new Date(formData.get('scheduledDate') as string)
        : undefined,
      sessionLocation: formData.get('sessionLocation') as string || undefined,
      requiresTravel: formData.get('requiresTravel') === 'true',
      travelDistance: formData.get('travelDistance')
        ? parseInt(formData.get('travelDistance') as string)
        : undefined,
      requiresAccommodation: formData.get('requiresAccommodation') === 'true',
      quotedAmount: formData.get('quotedAmount')
        ? parseFloat(formData.get('quotedAmount') as string)
        : undefined,
      depositAmount: formData.get('depositAmount')
        ? parseFloat(formData.get('depositAmount') as string)
        : undefined,
      privateNotes: formData.get('privateNotes') as string || undefined,
    };

    // Remove undefined values
    const cleanData = Object.fromEntries(
      Object.entries(data).filter(([_, v]) => v !== undefined)
    );

    const validated = updateCaseSchema.parse(cleanData);

    const caseRecord = await caseService.updateCase(caseId, user.id, validated);

    revalidatePath(`/mediator/cases/${caseId}`);
    revalidatePath('/mediator/cases');
    return { success: true, case: caseRecord };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update case' };
  }
}

/**
 * Update case status (mediator only)
 */
export async function updateCaseStatusAction(
  caseId: string,
  newStatus: CaseStatus,
  notes?: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR') {
      return { error: 'Unauthorized' };
    }

    const caseRecord = await caseService.updateCaseStatus(
      caseId,
      user.id,
      newStatus,
      notes
    );

    revalidatePath(`/mediator/cases/${caseId}`);
    revalidatePath('/mediator/cases');
    revalidatePath('/mediator/dashboard');
    return { success: true, case: caseRecord };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update status' };
  }
}

/**
 * Add note to case timeline
 */
export async function addCaseNoteAction(
  caseId: string,
  note: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user) {
      return { error: 'Unauthorized' };
    }

    const validated = addNoteSchema.parse({ note });

    await caseService.addCaseNote({
      caseId,
      userId: user.id,
      note: validated.note,
    });

    revalidatePath(`/mediator/cases/${caseId}`);
    revalidatePath(`/client/cases/${caseId}`);
    return { success: true };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to add note' };
  }
}

/**
 * Add party to case (mediator only)
 */
export async function addCasePartyAction(
  caseId: string,
  clientId: string,
  role: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR') {
      return { error: 'Unauthorized' };
    }

    const validated = addPartySchema.parse({ clientId, role });

    await caseService.addCaseParty(
      caseId,
      user.id,
      validated.clientId,
      validated.role
    );

    revalidatePath(`/mediator/cases/${caseId}`);
    return { success: true };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to add party' };
  }
}
