import { prisma } from '@/lib/db';
import { CaseStatus } from '@prisma/client';
import { NotFoundError, ForbiddenError, ValidationError } from '@/modules/shared/errors';
import { createAuditLog } from '@/modules/shared/audit';

export interface CreateInquiryData {
  mediatorId: string;
  clientId: string;
  title: string;
  description: string;
  disputeType?: string;
  estimatedHours?: number;
  requiresTravel?: boolean;
  travelDistance?: number;
  requiresAccommodation?: boolean;
}

export interface UpdateCaseData {
  title?: string;
  description?: string;
  disputeType?: string;
  estimatedHours?: number;
  scheduledDate?: Date;
  sessionLocation?: string;
  requiresTravel?: boolean;
  travelDistance?: number;
  requiresAccommodation?: boolean;
  quotedAmount?: number;
  depositAmount?: number;
  privateNotes?: string;
}

export interface AddCaseNoteData {
  caseId: string;
  userId: string;
  note: string;
}

export class CaseService {
  /**
   * Create a new inquiry (from public form or client)
   */
  async createInquiry(data: CreateInquiryData) {
    // Generate unique case number
    const caseNumber = await this.generateCaseNumber();

    // Create case with primary party
    const caseRecord = await prisma.case.create({
      data: {
        mediatorId: data.mediatorId,
        caseNumber,
        title: data.title,
        description: data.description,
        disputeType: data.disputeType,
        estimatedHours: data.estimatedHours,
        requiresTravel: data.requiresTravel || false,
        travelDistance: data.travelDistance,
        requiresAccommodation: data.requiresAccommodation || false,
        status: 'INQUIRY',
        parties: {
          create: {
            clientId: data.clientId,
            role: 'Primary Contact',
            isPrimary: true,
          },
        },
        timeline: {
          create: {
            event: 'Inquiry submitted',
            description: 'Client submitted inquiry',
            status: 'INQUIRY',
            performedBy: data.clientId,
          },
        },
      },
      include: {
        parties: {
          include: {
            client: {
              select: {
                firstName: true,
                lastName: true,
                email: true,
                phone: true,
              },
            },
          },
        },
        mediator: {
          select: {
            firstName: true,
            lastName: true,
            email: true,
          },
        },
      },
    });

    // Create audit log
    await createAuditLog({
      userId: data.clientId,
      action: 'CASE_CREATE',
      resource: 'case',
      resourceId: caseRecord.id,
      metadata: {
        caseNumber: caseRecord.caseNumber,
        mediatorId: data.mediatorId,
        status: 'INQUIRY',
      },
    });

    return caseRecord;
  }

  /**
   * Get case by ID with full relations
   */
  async getCaseById(caseId: string, userId: string) {
    const caseRecord = await prisma.case.findUnique({
      where: { id: caseId },
      include: {
        parties: {
          include: {
            client: {
              select: {
                id: true,
                userId: true,
                firstName: true,
                lastName: true,
                phone: true,
                address: true,
                user: {
                  select: {
                    email: true,
                  },
                },
              },
            },
          },
        },
        mediator: {
          select: {
            id: true,
            userId: true,
            firstName: true,
            lastName: true,
            email: true,
            phone: true,
          },
        },
        timeline: {
          orderBy: {
            createdAt: 'desc',
          },
        },
        documents: {
          orderBy: {
            createdAt: 'desc',
          },
        },
        payments: {
          orderBy: {
            createdAt: 'desc',
          },
        },
      },
    });

    if (!caseRecord) {
      throw new NotFoundError('Case not found');
    }

    // Check access permissions
    const isMediator = caseRecord.mediator.userId === userId;
    const isClient = caseRecord.parties.some(party => party.client.userId === userId);

    if (!isMediator && !isClient) {
      throw new ForbiddenError('You do not have access to this case');
    }

    return caseRecord;
  }

  /**
   * Get all cases for a mediator
   */
  async getMediatorCases(mediatorId: string, status?: CaseStatus) {
    return prisma.case.findMany({
      where: {
        mediatorId,
        ...(status && { status }),
      },
      include: {
        parties: {
          include: {
            client: {
              select: {
                firstName: true,
                lastName: true,
                user: {
                  select: {
                    email: true,
                  },
                },
              },
            },
          },
        },
        _count: {
          select: {
            documents: true,
            payments: true,
            timeline: true,
          },
        },
      },
      orderBy: {
        createdAt: 'desc',
      },
    });
  }

  /**
   * Get all cases for a client
   */
  async getClientCases(clientId: string, status?: CaseStatus) {
    return prisma.case.findMany({
      where: {
        parties: {
          some: { clientId },
        },
        ...(status && { status }),
      },
      include: {
        mediator: {
          select: {
            firstName: true,
            lastName: true,
            photoUrl: true,
            slug: true,
          },
        },
        parties: {
          where: { clientId },
          select: {
            role: true,
            isPrimary: true,
          },
        },
      },
      orderBy: {
        createdAt: 'desc',
      },
    });
  }

  /**
   * Update case details (mediator only)
   */
  async updateCase(
    caseId: string,
    userId: string,
    data: UpdateCaseData
  ) {
    const caseRecord = await this.getCaseById(caseId, userId);

    // Verify user is the mediator
    if (caseRecord.mediator.userId !== userId) {
      throw new ForbiddenError('Only the assigned mediator can update case details');
    }

    const updated = await prisma.case.update({
      where: { id: caseId },
      data: {
        ...data,
        updatedAt: new Date(),
      },
    });

    // Create audit log
    await createAuditLog({
      userId,
      action: 'CASE_UPDATE',
      resource: 'case',
      resourceId: caseId,
      metadata: {
        updatedFields: Object.keys(data),
        caseNumber: caseRecord.caseNumber,
      },
    });

    return updated;
  }

  /**
   * Update case status with timeline entry
   */
  async updateCaseStatus(
    caseId: string,
    userId: string,
    newStatus: CaseStatus,
    notes?: string
  ) {
    const caseRecord = await this.getCaseById(caseId, userId);

    // Verify user is the mediator
    if (caseRecord.mediator.userId !== userId) {
      throw new ForbiddenError('Only the assigned mediator can update case status');
    }

    // Validate status transition
    this.validateStatusTransition(caseRecord.status, newStatus);

    // Update case and create timeline entry
    const updated = await prisma.case.update({
      where: { id: caseId },
      data: {
        status: newStatus,
        ...(newStatus === 'COMPLETED' && { completedDate: new Date() }),
        timeline: {
          create: {
            event: `Status changed to ${newStatus}`,
            description: notes,
            status: newStatus,
            performedBy: userId,
          },
        },
      },
      include: {
        timeline: {
          orderBy: { createdAt: 'desc' },
          take: 1,
        },
      },
    });

    // Create audit log
    await createAuditLog({
      userId,
      action: 'CASE_STATUS_CHANGE',
      resource: 'case',
      resourceId: caseId,
      metadata: {
        caseNumber: caseRecord.caseNumber,
        oldStatus: caseRecord.status,
        newStatus,
        notes,
      },
    });

    return updated;
  }

  /**
   * Add a note to case timeline
   */
  async addCaseNote(data: AddCaseNoteData) {
    const caseRecord = await this.getCaseById(data.caseId, data.userId);

    // Verify user has access
    const isMediator = caseRecord.mediator.userId === data.userId;
    const isClient = caseRecord.parties.some(party => party.client.userId === data.userId);

    if (!isMediator && !isClient) {
      throw new ForbiddenError('You do not have access to this case');
    }

    const timeline = await prisma.caseTimeline.create({
      data: {
        caseId: data.caseId,
        event: 'Note added',
        description: data.note,
        performedBy: data.userId,
      },
    });

    // Create audit log
    await createAuditLog({
      userId: data.userId,
      action: 'CASE_UPDATE',
      resource: 'case',
      resourceId: data.caseId,
      metadata: {
        action: 'note_added',
        caseNumber: caseRecord.caseNumber,
      },
    });

    return timeline;
  }

  /**
   * Add additional party to case
   */
  async addCaseParty(
    caseId: string,
    userId: string,
    clientId: string,
    role: string
  ) {
    const caseRecord = await this.getCaseById(caseId, userId);

    // Verify user is the mediator
    if (caseRecord.mediator.userId !== userId) {
      throw new ForbiddenError('Only the assigned mediator can add parties');
    }

    // Check if party already exists
    const existingParty = await prisma.caseParty.findUnique({
      where: {
        caseId_clientId: {
          caseId,
          clientId,
        },
      },
    });

    if (existingParty) {
      throw new ValidationError('Client is already a party to this case');
    }

    const party = await prisma.caseParty.create({
      data: {
        caseId,
        clientId,
        role,
        isPrimary: false,
      },
      include: {
        client: true,
      },
    });

    // Add timeline entry
    await prisma.caseTimeline.create({
      data: {
        caseId,
        event: 'Party added',
        description: `${party.client.firstName} ${party.client.lastName} added as ${role}`,
        performedBy: userId,
      },
    });

    // Create audit log
    await createAuditLog({
      userId,
      action: 'CASE_UPDATE',
      resource: 'case',
      resourceId: caseId,
      metadata: {
        action: 'party_added',
        clientId,
        role,
        caseNumber: caseRecord.caseNumber,
      },
    });

    return party;
  }

  /**
   * Generate unique case number
   */
  private async generateCaseNumber(): Promise<string> {
    const year = new Date().getFullYear();
    const count = await prisma.case.count({
      where: {
        createdAt: {
          gte: new Date(`${year}-01-01`),
        },
      },
    });

    return `M8-${year}-${String(count + 1).padStart(4, '0')}`;
  }

  /**
   * Validate status transition
   */
  private validateStatusTransition(currentStatus: CaseStatus, newStatus: CaseStatus) {
    const validTransitions: Record<CaseStatus, CaseStatus[]> = {
      INQUIRY: ['QUOTE_SENT', 'CANCELLED'],
      QUOTE_SENT: ['BOOKED', 'CANCELLED'],
      BOOKED: ['IN_PROGRESS', 'CANCELLED'],
      IN_PROGRESS: ['COMPLETED', 'CANCELLED'],
      COMPLETED: ['CLOSED'],
      CLOSED: [],
      CANCELLED: [],
    };

    if (!validTransitions[currentStatus].includes(newStatus)) {
      throw new ValidationError(
        `Cannot transition from ${currentStatus} to ${newStatus}`
      );
    }
  }

  /**
   * Get case statistics for mediator
   */
  async getMediatorStats(mediatorId: string) {
    const [total, inquiries, active, completed] = await Promise.all([
      prisma.case.count({ where: { mediatorId } }),
      prisma.case.count({ where: { mediatorId, status: 'INQUIRY' } }),
      prisma.case.count({
        where: {
          mediatorId,
          status: { in: ['BOOKED', 'IN_PROGRESS'] },
        },
      }),
      prisma.case.count({ where: { mediatorId, status: 'COMPLETED' } }),
    ]);

    return {
      total,
      inquiries,
      active,
      completed,
    };
  }
}
