import { z } from 'zod';

export const createInquirySchema = z.object({
  mediatorId: z.string().min(1, 'Mediator is required'),
  title: z.string().min(5, 'Title must be at least 5 characters').max(200, 'Title too long'),
  description: z.string().min(20, 'Description must be at least 20 characters').max(2000, 'Description too long'),
  disputeType: z.string().optional(),
  estimatedHours: z.number().min(0).max(1000).optional(),
  requiresTravel: z.boolean().optional(),
  travelDistance: z.number().min(0).max(5000).optional(),
  requiresAccommodation: z.boolean().optional(),
});

export const updateCaseSchema = z.object({
  title: z.string().min(5).max(200).optional(),
  description: z.string().min(20).max(2000).optional(),
  disputeType: z.string().optional(),
  estimatedHours: z.number().min(0).max(1000).optional(),
  scheduledDate: z.date().optional(),
  sessionLocation: z.string().max(500).optional(),
  requiresTravel: z.boolean().optional(),
  travelDistance: z.number().min(0).max(5000).optional(),
  requiresAccommodation: z.boolean().optional(),
  quotedAmount: z.number().min(0).max(1000000).optional(),
  depositAmount: z.number().min(0).max(1000000).optional(),
  privateNotes: z.string().max(5000).optional(),
});

export const addNoteSchema = z.object({
  note: z.string().min(1, 'Note cannot be empty').max(2000, 'Note too long'),
});

export const addPartySchema = z.object({
  clientId: z.string().min(1, 'Client is required'),
  role: z.string().min(1, 'Role is required').max(100, 'Role too long'),
});

// Common dispute types
export const DISPUTE_TYPES = [
  'Family Dispute',
  'Divorce',
  'Property Settlement',
  'Workplace Dispute',
  'Commercial Dispute',
  'Business Partnership',
  'Neighbourhood Dispute',
  'Estate Dispute',
  'Consumer Dispute',
  'Other',
] as const;

export type CreateInquiryInput = z.infer<typeof createInquirySchema>;
export type UpdateCaseInput = z.infer<typeof updateCaseSchema>;
export type AddNoteInput = z.infer<typeof addNoteSchema>;
export type AddPartyInput = z.infer<typeof addPartySchema>;
