'use server';

import { getCurrentUser } from '@/modules/auth/actions';
import { HubSpotIntegrationService } from './service';

const hubspotService = new HubSpotIntegrationService();

interface ActionResult {
  error?: string;
  success?: boolean;
  data?: any;
}

/**
 * Test HubSpot connection
 */
export async function testHubSpotConnectionAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    if (!hubspotService.isEnabled()) {
      return { 
        success: false, 
        data: { 
          message: 'HubSpot integration is disabled. Set HUBSPOT_ENABLED=true in environment variables.' 
        } 
      };
    }

    const connected = await hubspotService.testConnection();

    return {
      success: true,
      data: {
        connected,
        message: connected
          ? 'Successfully connected to HubSpot'
          : 'Failed to connect to HubSpot. Check API key.',
      },
    };
  } catch (error) {
    return {
      error: error instanceof Error ? error.message : 'Connection test failed',
    };
  }
}

/**
 * Manually sync a case to HubSpot
 */
export async function syncCaseToHubSpotAction(caseId: string): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || (user.role !== 'ADMIN' && user.role !== 'MEDIATOR')) {
      return { error: 'Unauthorized' };
    }

    if (!hubspotService.isEnabled()) {
      return { error: 'HubSpot integration is disabled' };
    }

    await hubspotService.syncCaseCreated(caseId);

    return {
      success: true,
      data: { message: 'Case synced to HubSpot' },
    };
  } catch (error) {
    return {
      error: error instanceof Error ? error.message : 'Sync failed',
    };
  }
}

/**
 * Get HubSpot integration status
 */
export async function getHubSpotStatusAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    return {
      success: true,
      data: {
        enabled: hubspotService.isEnabled(),
        hasApiKey: !!process.env.HUBSPOT_API_KEY,
      },
    };
  } catch (error) {
    return { error: 'Failed to get status' };
  }
}
