import { prisma } from '@/lib/db';
import { getHubSpotClient, HUBSPOT_DEAL_STAGES } from '@/lib/hubspot';
import { createAuditLog } from '@/modules/shared/audit';

export class HubSpotIntegrationService {
  private hubspot = getHubSpotClient();

  /**
   * Check if HubSpot integration is enabled
   */
  isEnabled(): boolean {
    return this.hubspot.isEnabled();
  }

  /**
   * Sync case to HubSpot on creation (new enquiry)
   * Creates contact for client and deal for case
   */
  async syncCaseCreated(caseId: string): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
        include: {
          mediator: true,
          parties: {
            include: {
              client: {
                include: {
                  user: true,
                },
              },
            },
          },
        },
      });

      if (!caseRecord) {
        console.error('[HubSpot] Case not found:', caseId);
        return;
      }

      // Get primary party (first party)
      const primaryParty = caseRecord.parties.find(p => p.isPrimary) || caseRecord.parties[0];
      if (!primaryParty) {
        console.error('[HubSpot] No parties found for case:', caseId);
        return;
      }

      const client = primaryParty.client;

      // Create/update contact
      const contactId = await this.hubspot.upsertContact({
        email: client.user.email,
        firstname: client.firstName,
        lastname: client.lastName,
        phone: client.phone || undefined,
        lifecyclestage: 'lead',
      });

      if (!contactId) {
        console.error('[HubSpot] Failed to create contact for case:', caseId);
        return;
      }

      // Create deal
      const dealName = `${caseRecord.caseNumber} - ${client.firstName} ${client.lastName}`;
      const dealId = await this.hubspot.createDeal(
        {
          dealname: dealName,
          amount: caseRecord.quotedAmount ? Number(caseRecord.quotedAmount) : undefined,
          dealstage: HUBSPOT_DEAL_STAGES.NEW_ENQUIRY,
          closedate: caseRecord.createdAt.toISOString(),
        },
        contactId
      );

      if (!dealId) {
        console.error('[HubSpot] Failed to create deal for case:', caseId);
        return;
      }

      // Store HubSpot IDs in case metadata
      await prisma.case.update({
        where: { id: caseId },
        data: {
          metadata: {
            hubspotDealId: dealId,
            hubspotContactId: contactId,
            hubspotSyncedAt: new Date().toISOString(),
          } as any,
        },
      });

      await createAuditLog({
        action: 'CASE_UPDATE',
        resource: 'case',
        resourceId: caseId,
        metadata: {
          action: 'hubspot_sync',
          dealId,
          contactId,
        },
      });

      console.log(`[HubSpot] Synced case ${caseRecord.caseNumber} - Deal: ${dealId}, Contact: ${contactId}`);
    } catch (error) {
      console.error('[HubSpot] Error syncing case created:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Update deal stage when deposit is paid
   */
  async syncDepositPaid(caseId: string): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
      });

      if (!caseRecord || !caseRecord.metadata) {
        return;
      }

      const metadata = caseRecord.metadata as any;
      const dealId = metadata.hubspotDealId;

      if (!dealId) {
        console.log('[HubSpot] No deal ID found for case:', caseId);
        return;
      }

      // Update deal stage
      const success = await this.hubspot.updateDealStage(
        dealId,
        HUBSPOT_DEAL_STAGES.DEPOSIT_PAID
      );

      if (success) {
        // Also update amount if available
        if (caseRecord.depositAmount) {
          await this.hubspot.updateDeal(dealId, {
            amount: Number(caseRecord.depositAmount),
          });
        }

        console.log(`[HubSpot] Updated deal ${dealId} to DEPOSIT_PAID`);
      }
    } catch (error) {
      console.error('[HubSpot] Error syncing deposit paid:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Update deal stage when case is approved to proceed
   */
  async syncApprovedToProceed(caseId: string): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
      });

      if (!caseRecord || !caseRecord.metadata) {
        return;
      }

      const metadata = caseRecord.metadata as any;
      const dealId = metadata.hubspotDealId;

      if (!dealId) {
        return;
      }

      const success = await this.hubspot.updateDealStage(
        dealId,
        HUBSPOT_DEAL_STAGES.APPROVED_TO_PROCEED
      );

      if (success) {
        console.log(`[HubSpot] Updated deal ${dealId} to APPROVED_TO_PROCEED`);
      }
    } catch (error) {
      console.error('[HubSpot] Error syncing approved to proceed:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Update deal stage when balance is paid
   */
  async syncBalancePaid(caseId: string, totalPaid: number): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
      });

      if (!caseRecord || !caseRecord.metadata) {
        return;
      }

      const metadata = caseRecord.metadata as any;
      const dealId = metadata.hubspotDealId;

      if (!dealId) {
        return;
      }

      const success = await this.hubspot.updateDealStage(
        dealId,
        HUBSPOT_DEAL_STAGES.BALANCE_PAID
      );

      if (success) {
        // Update deal amount with total paid
        await this.hubspot.updateDeal(dealId, {
          amount: totalPaid,
        });

        console.log(`[HubSpot] Updated deal ${dealId} to BALANCE_PAID with amount: $${totalPaid}`);
      }
    } catch (error) {
      console.error('[HubSpot] Error syncing balance paid:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Update deal stage when case is completed
   */
  async syncCaseCompleted(caseId: string): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
        include: {
          payments: {
            where: {
              status: 'SUCCEEDED',
            },
          },
        },
      });

      if (!caseRecord || !caseRecord.metadata) {
        return;
      }

      const metadata = caseRecord.metadata as any;
      const dealId = metadata.hubspotDealId;

      if (!dealId) {
        return;
      }

      // Calculate total paid
      const totalPaid = caseRecord.payments.reduce(
        (sum, payment) => sum + Number(payment.stripeGrossAmount),
        0
      );

      const success = await this.hubspot.updateDealStage(
        dealId,
        HUBSPOT_DEAL_STAGES.COMPLETED
      );

      if (success) {
        await this.hubspot.updateDeal(dealId, {
          amount: totalPaid,
          closedate: new Date().toISOString(),
        });

        console.log(`[HubSpot] Updated deal ${dealId} to COMPLETED`);
      }
    } catch (error) {
      console.error('[HubSpot] Error syncing case completed:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Update deal stage when case is cancelled
   */
  async syncCaseCancelled(caseId: string, reason?: string): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
      });

      if (!caseRecord || !caseRecord.metadata) {
        return;
      }

      const metadata = caseRecord.metadata as any;
      const dealId = metadata.hubspotDealId;
      const contactId = metadata.hubspotContactId;

      if (!dealId) {
        return;
      }

      const success = await this.hubspot.updateDealStage(
        dealId,
        HUBSPOT_DEAL_STAGES.CANCELLED
      );

      if (success) {
        // Add note with cancellation reason
        if (contactId && reason) {
          await this.hubspot.addNoteToContact(
            contactId,
            `Case cancelled: ${reason}`
          );
        }

        console.log(`[HubSpot] Updated deal ${dealId} to CANCELLED`);
      }
    } catch (error) {
      console.error('[HubSpot] Error syncing case cancelled:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Add activity note to HubSpot contact
   */
  async addActivityNote(caseId: string, activity: string): Promise<void> {
    if (!this.isEnabled()) {
      return;
    }

    try {
      const caseRecord = await prisma.case.findUnique({
        where: { id: caseId },
      });

      if (!caseRecord || !caseRecord.metadata) {
        return;
      }

      const metadata = caseRecord.metadata as any;
      const contactId = metadata.hubspotContactId;

      if (!contactId) {
        return;
      }

      await this.hubspot.addNoteToContact(contactId, activity);
      console.log(`[HubSpot] Added activity note to contact ${contactId}`);
    } catch (error) {
      console.error('[HubSpot] Error adding activity note:', error);
      // Non-blocking - don't throw
    }
  }

  /**
   * Test HubSpot connection
   */
  async testConnection(): Promise<boolean> {
    return this.hubspot.testConnection();
  }
}
