'use server';

import { revalidatePath } from 'next/cache';
import { PricingService } from './service';
import { pricingSchema } from './validation';
import { getCurrentUser } from '@/modules/auth/actions';
import { ZodError } from 'zod';

const pricingService = new PricingService();

interface ActionResult {
  error?: string;
  success?: boolean;
  pricing?: any;
}

/**
 * Update mediator pricing
 */
export async function updatePricingAction(formData: FormData): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const data = {
      preMediationFee: formData.get('preMediationFee') 
        ? parseFloat(formData.get('preMediationFee') as string)
        : undefined,
      hourlyRate: parseFloat(formData.get('hourlyRate') as string),
      hourlyRateBasis: formData.get('hourlyRateBasis') as 'PER_PARTY' | 'TOTAL',
      minimumHours: formData.get('minimumHours')
        ? parseFloat(formData.get('minimumHours') as string)
        : undefined,
      travelIncludedKm: formData.get('travelIncludedKm')
        ? parseInt(formData.get('travelIncludedKm') as string)
        : undefined,
      travelRatePerKm: formData.get('travelRatePerKm')
        ? parseFloat(formData.get('travelRatePerKm') as string)
        : undefined,
      accommodationPolicy: formData.get('accommodationPolicy') as string || undefined,
      accommodationRate: formData.get('accommodationRate')
        ? parseFloat(formData.get('accommodationRate') as string)
        : undefined,
      extraCostsNotes: formData.get('extraCostsNotes') as string || undefined,
    };

    const validated = pricingSchema.parse(data);

    const pricing = await pricingService.upsertPricing(
      user.mediator.id,
      user.id,
      validated
    );

    revalidatePath('/mediator/profile/pricing');
    return { success: true, pricing };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update pricing' };
  }
}
