import { prisma } from '@/lib/db';
import { HourlyRateBasis } from '@prisma/client';
import { NotFoundError, ForbiddenError } from '@/modules/shared/errors';
import { createAuditLog } from '@/modules/shared/audit';

export interface PricingData {
  preMediationFee?: number;
  hourlyRate: number;
  hourlyRateBasis: HourlyRateBasis;
  minimumHours?: number;
  travelIncludedKm?: number;
  travelRatePerKm?: number;
  accommodationPolicy?: string;
  accommodationRate?: number;
  extraCostsNotes?: string;
}

export class PricingService {
  /**
   * Get pricing for a mediator
   */
  async getPricing(mediatorId: string) {
    const pricing = await prisma.mediatorPricing.findUnique({
      where: { mediatorId },
    });

    return pricing;
  }

  /**
   * Create or update pricing
   */
  async upsertPricing(
    mediatorId: string,
    userId: string,
    data: PricingData
  ) {
    // Verify mediator exists and user owns it
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator profile not found');
    }

    if (mediator.userId !== userId) {
      throw new ForbiddenError('You do not have permission to update this pricing');
    }

    // Upsert pricing
    const pricing = await prisma.mediatorPricing.upsert({
      where: { mediatorId },
      create: {
        mediatorId,
        ...data,
      },
      update: {
        ...data,
        updatedAt: new Date(),
      },
    });

    // Create audit log
    await createAuditLog({
      userId,
      action: 'PROFILE_UPDATE',
      resource: 'mediator_pricing',
      resourceId: pricing.id,
      metadata: { mediatorId },
    });

    return pricing;
  }

  /**
   * Calculate a quote based on case parameters
   */
  async calculateQuote(params: {
    mediatorId: string;
    estimatedHours: number;
    numberOfParties: number;
    requiresTravel: boolean;
    travelDistance?: number;
    requiresAccommodation: boolean;
  }) {
    const pricing = await this.getPricing(params.mediatorId);

    if (!pricing) {
      throw new NotFoundError('Pricing not configured for this mediator');
    }

    const mediator = await prisma.mediator.findUnique({
      where: { id: params.mediatorId },
      select: { isGstRegistered: true },
    });

    let subtotal = 0;

    // Pre-mediation fee
    const preMediationFee = pricing.preMediationFee?.toNumber() || 0;
    subtotal += preMediationFee;

    // Hourly charges
    const hourlyRate = pricing.hourlyRate.toNumber();
    let hourlyTotal = 0;

    if (pricing.hourlyRateBasis === 'PER_PARTY') {
      hourlyTotal = hourlyRate * params.estimatedHours * params.numberOfParties;
    } else {
      hourlyTotal = hourlyRate * params.estimatedHours;
    }

    subtotal += hourlyTotal;

    // Travel costs
    let travelCost = 0;
    if (params.requiresTravel && params.travelDistance) {
      const includedKm = pricing.travelIncludedKm || 0;
      const extraKm = Math.max(0, params.travelDistance - includedKm);
      const ratePerKm = pricing.travelRatePerKm?.toNumber() || 0;
      travelCost = extraKm * ratePerKm;
      subtotal += travelCost;
    }

    // Accommodation
    let accommodationCost = 0;
    if (params.requiresAccommodation && pricing.accommodationRate) {
      accommodationCost = pricing.accommodationRate.toNumber();
      subtotal += accommodationCost;
    }

    // GST calculation (if mediator is registered)
    let gstAmount = 0;
    let totalIncGst = subtotal;

    if (mediator?.isGstRegistered) {
      gstAmount = subtotal * 0.10; // 10% GST
      totalIncGst = subtotal + gstAmount;
    }

    return {
      subtotal,
      breakdown: {
        preMediationFee,
        hourlyCharges: hourlyTotal,
        travelCost,
        accommodationCost,
      },
      gstAmount,
      totalIncGst,
      isGstRegistered: mediator?.isGstRegistered || false,
      hourlyRateBasis: pricing.hourlyRateBasis,
    };
  }
}
