'use server';

import { redirect } from 'next/navigation';
import { revalidatePath } from 'next/cache';
import { MediatorProfileService } from './service';
import { 
  updateProfileSchema, 
  specializationsSchema, 
  certificationsSchema,
  languagesSchema 
} from './validation';
import { getCurrentUser } from '@/modules/auth/actions';
import { ZodError } from 'zod';
import { sendEmail } from '@/lib/email';

const profileService = new MediatorProfileService();

interface ActionResult {
  error?: string;
  success?: boolean;
  mediator?: any;
}

/**
 * Update basic profile information
 */
export async function updateProfileAction(formData: FormData): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const data = {
      firstName: formData.get('firstName') as string || undefined,
      lastName: formData.get('lastName') as string || undefined,
      displayName: formData.get('displayName') as string || undefined,
      bio: formData.get('bio') as string || undefined,
      phone: formData.get('phone') as string || undefined,
      officeAddress: formData.get('officeAddress') as string || undefined,
      yearsExperience: formData.get('yearsExperience') 
        ? parseInt(formData.get('yearsExperience') as string) 
        : undefined,
    };

    const validated = updateProfileSchema.parse(data);

    const mediator = await profileService.updateProfile(
      user.mediator.id,
      user.id,
      validated
    );

    revalidatePath('/mediator/profile');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update profile' };
  }
}

/**
 * Update specializations
 */
export async function updateSpecializationsAction(
  specializations: string[]
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const validated = specializationsSchema.parse({ specializations });

    const mediator = await profileService.updateProfile(
      user.mediator.id,
      user.id,
      { specializations: validated.specializations }
    );

    revalidatePath('/mediator/profile');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update specializations' };
  }
}

/**
 * Update certifications
 */
export async function updateCertificationsAction(
  certifications: string[]
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const validated = certificationsSchema.parse({ certifications });

    const mediator = await profileService.updateProfile(
      user.mediator.id,
      user.id,
      { certifications: validated.certifications }
    );

    revalidatePath('/mediator/profile');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update certifications' };
  }
}

/**
 * Update languages
 */
export async function updateLanguagesAction(
  languages: string[]
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const validated = languagesSchema.parse({ languages });

    const mediator = await profileService.updateProfile(
      user.mediator.id,
      user.id,
      { languages: validated.languages }
    );

    revalidatePath('/mediator/profile');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof ZodError) {
      return { error: error.errors[0].message };
    }
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to update languages' };
  }
}

/**
 * Submit profile for approval
 */
export async function submitForApprovalAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const mediator = await profileService.submitForApproval(
      user.mediator.id,
      user.id
    );

    revalidatePath('/mediator/profile');
    revalidatePath('/mediator/dashboard');
    
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to submit profile for approval' };
  }
}

/**
 * Admin: Approve mediator
 */
export async function approveMediatorAction(
  mediatorId: string,
  notes?: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const mediator = await profileService.approveMediator(
      mediatorId,
      user.id,
      notes
    );

    // Send approval email
    await sendEmail({
      to: mediator.user.email,
      template: 'MEDIATOR_APPROVED',
      data: {
        firstName: mediator.firstName,
        slug: mediator.slug,
      },
    });

    revalidatePath('/admin/approvals');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to approve mediator' };
  }
}

/**
 * Admin: Reject mediator
 */
export async function rejectMediatorAction(
  mediatorId: string,
  reason: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const mediator = await profileService.rejectMediator(
      mediatorId,
      user.id,
      reason
    );

    // Send rejection email
    await sendEmail({
      to: mediator.user.email,
      template: 'MEDIATOR_REJECTED',
      data: {
        firstName: mediator.firstName,
        reason,
      },
    });

    revalidatePath('/admin/approvals');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to reject mediator' };
  }
}

/**
 * Admin: Publish mediator
 */
export async function publishMediatorAction(mediatorId: string): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const mediator = await profileService.publishMediator(mediatorId, user.id);

    revalidatePath('/admin/approvals');
    revalidatePath('/practices');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to publish mediator' };
  }
}

/**
 * Admin: Suspend mediator
 */
export async function suspendMediatorAction(
  mediatorId: string,
  reason: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const mediator = await profileService.suspendMediator(
      mediatorId,
      user.id,
      reason
    );

    revalidatePath('/admin/approvals');
    revalidatePath('/practices');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to suspend mediator' };
  }
}

/**
 * Admin: Unpublish mediator
 */
export async function unpublishMediatorAction(mediatorId: string): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const mediator = await profileService.unpublishMediator(mediatorId, user.id);

    revalidatePath('/admin/approvals');
    revalidatePath('/practices');
    return { success: true, mediator };
  } catch (error) {
    if (error instanceof Error) {
      return { error: error.message };
    }
    return { error: 'Failed to unpublish mediator' };
  }
}
